(ns {{ name }}.core
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            [clojure.tools.reader :as reader]

            carica.core
            [com.stuartsierra.component :as component :refer [using]]
            {{#postgres}}
            [hikari-cp.core :as hikari]
            {{/postgres}}
            {{#web}}
            http-server
            {{/web}}

            [{{ name }}.utils :as utils]{{#web}}
            [{{ name }}.web :as web]{{/web}})
  (:gen-class))
{{#postgres}}

(defrecord DbPool [config datasource]
  component/Lifecycle
  (start [c]
    (if (not datasource)
      (assoc c :datasource (hikari/make-datasource config))
      c))
  (stop [c]
    (when datasource
      (hikari/close-datasource datasource))
    (assoc c :datasource nil)))

(defn db-pool [config]
  (->DbPool config nil))
{{/postgres}}

(defn system-component
  [{:keys [{{#postgres}}db{{/postgres}}{{#web}} web{{/web}}] :as config}]
  (component/system-map
    {{#postgres}}
    :db (db-pool db)
    {{/postgres}}
    {{#web}}
    :resources (using {} [{{#postgres}}:db{{/postgres}}])
    :http-server (using (http-server/http-server web web/build-handler)
                        [:resources])
    {{/web}}
    :loaded-config config))

(def parsers
  {:string identity
   :exists boolean
   :integer #(Integer/parseInt %)
   :long #(Long/parseLong %)
   :float #(Float/parseFloat %)
   :double #(Double/parseDouble %)})

(defn no-parser-found [k]
  (fn [v]
    (throw (ex-info "No env parser found for key" {:key k :to-parse v}))))

(defn env [k]
  (let [k (if (string? k) [k :string] k)
        [k t] k]
    ((get parsers t (no-parser-found t))
     (System/getenv k))))

(defn load-config [& paths]
  (binding [reader/*data-readers* (assoc reader/*data-readers* 'env #'{{ name }}.core/env)]
    (->> paths
         (map #(io/file %))
         (filter #(.exists ^java.io.File %))
         carica.core/configurer
         .invoke)))

(def system nil)

(defn -main
  ([] (-main "config/config-dev.clj" "config/config-dev.sample.clj"))
  ([& config-paths]
   (log/info "system starting")
   (let [sys (-> (apply load-config config-paths)
                 system-component
                 component/start)]
     (alter-var-root #'system (constantly sys))
     (utils/add-shutdown-hook! #(do (log/info "system stoping")
                                    (component/stop sys)
                                    (log/info "system stopped")))
     (log/info "system started")
     (.join ^Thread (Thread/currentThread)))))
