(ns reify.bslite.util
  (:require
    [schema.core :as s]
    [clojure.string :refer [join]]))

(s/defschema VTree s/Any)

(s/defschema ClassnameSpec
  "Classnames can arise from a number of different Clojure types. For
  convenience, keywords, strings, symbols, vectors, and maps from any of the
  above to boolean values are all accepted."
  (s/either
    s/Keyword
    s/Str
    s/Symbol
    [(s/recursive #'ClassnameSpec)]
    {(s/recursive #'ClassnameSpec) s/Bool}))

(defprotocol ^:private ClassNameable
  (-classnames
    [it] "Produce a sequence of classnames arising from a given type."))

(extend-protocol ClassNameable
  #?(:clj clojure.lang.Keyword :cljs cljs.core.Keyword)
  (-classnames [s] [(name s)])

  #?(:clj java.lang.String :cljs string)
  (-classnames [s] [s])

  #?(:clj clojure.lang.Symbol :cljs cljs.core.Symbol)
  (-classnames [s] [(name s)])

  #?@(:cljs [cljs.core.List
             (-classnames [l] (mapcat -classnames l))])

  #?@(:clj [clojure.lang.LazilyPersistentVector
            (-classnames [v] (mapcat -classnames v))])

  #?(:clj  clojure.lang.PersistentVector
     :cljs cljs.core.PersistentVector)
  (-classnames [v] (mapcat -classnames v))

  #?@(:clj [clojure.lang.PersistentVector$ChunkedSeq
            (-classnames [v] (mapcat -classnames v))])

  #?(:clj  clojure.lang.LazySeq
     :cljs cljs.core.LazySeq)
  (-classnames [s] (mapcat -classnames s))


  #?(:clj  clojure.lang.PersistentHashSet
     :cljs cljs.core.PersistentHashSet)
  (-classnames [s] (mapcat -classnames s))

  #?(:clj  clojure.lang.PersistentHashMap
     :cljs cljs.core.PersistentHashMap)
  (-classnames [m]
    (mapcat -classnames (for [[k v] m :when v] k)))

  #?(:clj  clojure.lang.PersistentArrayMap
     :cljs cljs.core.PersistentArrayMap)
  (-classnames [m]
    (mapcat -classnames (for [[k v] m :when v] k))))

(s/defn classnames :- s/Str
  "Convert a dictionary of booleans into an HTML class list string."
  [& specs :- [ClassnameSpec]]
  (join " " (apply hash-set (mapcat -classnames specs))))

(s/defn label :- [VTree]
  "A simple hack to provide keys to all values in the Nav. Without them React
  complains quite a lot."
  [children :- [VTree]]
  (map-indexed #(if (coll? %2) (with-meta %2 {:key %1}) %2) children))

