(ns reify.tokamak.impl.context
  "Reagent components which manipulate React parent->child context. These can
   be used to establish a sort of DOM-local reader monad or localized global
   state effect which is useful for writing testable, convenient frontend code."
  (:require
    [reify.tokamak.impl.util :refer [prop-type-any shallow-clj->js]]
    [reagent.core :as r]))

(defn stash-context
  "Produce a DOM-free component stashing context available to all children.
  Context values are passed *through* the virtual dom tree providing a local,
  globally scoped namespace. Context values will be available in subcomponents
  formed with `given-context` or `with-context` so long as they reference the
  same key."
  [context child]
  (let [context-types (into {} (map (fn [[k _]] [k prop-type-any]) context))]
    (with-meta
      (fn [] child)
      {:child-context-types (shallow-clj->js context-types)
       :get-child-context   (fn [] (shallow-clj->js context))})))

(defn given-context
  "Produces a DOM-free component retrieving context (if available). First
  argument is a vector of keys, the values of which will be passed to the
  function in the second argument."
  [keys k]
  (let [context-types (into {} (map (fn [k] [k prop-type-any]) keys))]
    (with-meta
      (fn []
        (let [this (r/current-component)
              context (.-context this)
              values (map (fn [k] (aget context (name k))) keys)]
          (apply k values)))
      {:context-types (shallow-clj->js context-types)})))