(ns reify.tokamak.reactor
  "A Reactor is a *running* core state machine for a Reagent application.
   Build it from a Machine."
  (:require-macros [cljs.core.async.macros :refer [go-loop alt!]])
  (:require
    [reify.tokamak.view :as view]
    [reify.tokamak.protocols :as p]

    [reify.tokamak.machine :as m]
    [reify.tokamak.schemata :as scm]

    [schema.core :as s :include-macros true]

    [reagent.core :as r]
    [cljs.core.async :refer [timeout <! put! chan]]))

(deftype Reactor [root action-chan kill-fn]
  IDeref
  (-deref [rx] (deref (.-root rx))))

(s/defschema ReactorOptions
  {:middlewares [(s/=> scm/Middleware scm/MiddlewareAPI)]
   :action-chan scm/Channel
   :root        scm/Ratom
   :updater     (s/=> scm/Nil scm/State scm/State (s/=> scm/Nil scm/State))})

(s/defn build-default-reactor-options :- ReactorOptions
  []
  {:middlewares []
   :action-chan (chan)
   :root        nil
   :updater     (fn [_old-state new-state commit!] (commit! new-state))})

(defn- react!
  [control-chan action-chan handler]
  (go-loop []
    (alt!
      :priority true

      ;; If we're getting events off the control channel we'll die immediately
      control-chan nil

      ;; If we've recieved an action, we'll handle it carefully within a try
      ;; block and then recur.
      action-chan
      ([action]
        ;; (1) We sleep for a moment letting the main thread resume. If the action
        ;; metadata calls for an explicit repaint then we force that and give extra
        ;; time for the repaint
        (if (:flush-dom (meta action))
          (do (r/flush) (<! (timeout 20)))
          (<! (timeout 0)))
        ;; (2) We try the action catching errors if they occur. Exceptions kill the
        ;; current action queue and continue upward. The queue itself is restarted.
        (try (handler action)
             (catch js/Object ex
               (do #_(purge-chan action-chan)
                 (react! control-chan action-chan handler)
                 (throw ex))))
        ;; (3) Then we recur!
        (recur)))))

(s/defn make :- Reactor
  "Construct a live Reactor from a static Machine. Optionally pass an options
   map which describes a set of middleware to apply (left-to-right), the
   updater function with a signature like `(fn [old-state new-state commit!]
   ...)`, a state ratom to use instead of creating a fresh one, and an event
   channel to use instead of creating a fresh one.

   Generally the reactor behavior is extended using both middlewares and
   custom updater functions---middlewares enable custom logic around the
   interpretation and effect of actions while the updater allows last minute
   state update manipulation like schema validation or history record-keeping."
  ([machine] (make machine {}))
  ([machine :- (s/protocol p/IMachine)
    options :- ReactorOptions]
    (let [options (merge (build-default-reactor-options) options)

          {:keys [middlewares action-chan updater]} options

          control-chan (chan)

          root (or (:root options)
                   (r/atom (p/initial-state machine)))

          reducer (p/reducer machine)

          ;; This is the very core handler effect: it applies the reducer
          ;; and updates the root
          handler0 (fn handle-action [action]
                     (let [old-state @root
                           new-state (reducer action old-state)]
                       (updater old-state new-state (partial reset! root))))

          middleware-api {:dispatch (fn dispatch-async [a] (put! action-chan a))
                          :view     (fn view-state
                                      ([v] (view/view v @root))
                                      ([v d] (view/view v @root d)))}

          middlewares (map (fn [m] (m middleware-api)) middlewares)

          handler (reduce (fn [h m] (m h)) handler0 middlewares)]

      (react! control-chan action-chan handler)
      (Reactor. root action-chan #(put! control-chan ::stop)))))

(s/defn dispatch! :- scm/Nil
  "Send an action through a reactor."
  [rx :- Reactor, ev :- scm/Action]
  (put! (.-action-chan rx) ev)
  nil)

(s/defn stop! :- scm/Nil
  "Stop a running reactor."
  [rx :- Reactor]
  (.kill-fn rx))