(ns reify.tokamak.view
  "A view is a static depiction of a subpart of a value---it is similar to a
   path into a generic type, a zipper, or a lens."
  (:require
    [reify.tokamak.protocols :refer [IView -view]]))

(defn view
  "*Perform* a view, returning its values."
  ([v s] (view v s nil))
  ([v s default]
   (cond
     #?(:cljs (satisfies? IDeref v)
        :clj (instance? clojure.lang.IDeref v))
     (-view @v s default)

     :else
     (-view v s default))))

;; A Lookup type wraps a value meant explicitly to act as a key in a lookup
;; on the state
(deftype Lookup [key]

  IView
  (-view [l s d] (get s key d)))

(defn lookup
  "Construct a view from any value which can serve as a hash key which
   functions by looking up that key. Allows one to override default IView
   behavior of compound keys."
  [k] (->Lookup k))

(extend-protocol IView

  nil (-view [_ _ _] nil)

  #?@(:cljs [js/RegExp (-view [r s _] (re-matches r s))
             number (-view [n s _] (get s n))
             string (-view [n s _] (get s n))
             function (-view [f s _] (f s))]

      :clj  [Long (-view [n s _] (get s n))
             String (-view [n s _] (get s n))
             clojure.lang.IFn (-view [f s _] (f s))])

  #?(:cljs Keyword :clj clojure.lang.Keyword)
  (-view [k s d] (get s k d))

  #?(:cljs PersistentArrayMap :clj clojure.lang.PersistentArrayMap)
  (-view [m s d]
    (let [sentinel (gensym)
          seq (map (fn [[k v]] [k (view v s sentinel)]) m)
          fail? (some (partial = sentinel) (map second seq))]
      (if (not fail?)
        (into {} seq)
        d)))

  #?(:cljs PersistentVector :clj clojure.lang.PersistentVector)
  (-view [v s d]
    (let [sentinel (gensym)
          result (reduce
                   (fn [s e]
                     (if (= s sentinel) (reduced s)
                                        (view e s sentinel)))
                   s v)]
      (if (= result sentinel) d result)))

  #?(:cljs List :clj clojure.lang.PersistentList)
  (-view [v s d]
    (let [sentinel (gensym)
          result (reduce
                   (fn [s e]
                     (if (= s sentinel) (reduced s)
                                        (view e s sentinel)))
                   s v)]
      (if (= result sentinel) d result))))
