(ns clj-ftp-client.core
  "A org.apache.commons.net.ftp wrapper

  Most file manipulation function names originate from
  ordinary FTP commands: get, put, ls, cd, pwd, mkdir, rm and mv.
  The `mkdirs' function originates from java.io.File#mkdirs working like
  `mkdir -p'.

  Functions which return stream originate from method of
  org.apache.commons.net.ftp.FTPClient."
  (:refer-clojure :exclude [get])
  (:require [clojure.java.io :as io]
            [clojure.string :as str])
  (:import (org.apache.commons.net.ftp FTPClient FTPReply)
           (java.io InputStream OutputStream)
           (clj-ftp-client FTP FTPS)))

(defn ^FTPClient open
  "Retuns connected ftp(or ftps) handler Object that extends
  org.apache.commons.net.ftp.FTP implements java.io.Closeable.
  So you can use this with `with-open`(recommend).

  NOTICE:
    via http://wiki.apache.org/commons/Net/FrequentlyAskedQuestions
  If you want to share a single FTPClient instance between multiple
  threads, you must serialize access to the object with critical
  sections."
  [host & {:keys [protocol port user password file-type mode
                  data-timeout-ms
                  control-keep-alive-timeout-sec
                  control-keep-alive-reply-timeout-ms
                  list-hidden-files? ;=> default: server depended
                  ;; auto-detect-utf8?
                  ;; enable-passive-nat-workaround?
                  ;; enabled-remote-verification?
                  ]
           :or {protocol :ftp
                mode :passive
                file-type :binary
                user "anonymous"
                password ""
                data-timeout-ms 0
                control-keep-alive-timeout-sec 300
                control-keep-alive-reply-timeout-ms 1000
                ;; enable-passive-nat-workaround? true
                ;; enabled-remote-verification? true
                }}]
  {:pre [(#{:ftp :ftps} protocol)
         (or (not port) (integer? port))
         (#{:binary :ascii} file-type)
         (#{:passive :active} mode)]}
  (let [client (case protocol
                 :ftp  (clj-ftp-client.FTP.)
                 :ftps (clj-ftp-client.FTPS.))]
    ;; configurations before the connect.
    #_(doto client
        (.setAutodetectUTF8 client auto-detect-utf8?))
    (.connect client host (or port (.getDefaultPort client)))
    (let [reply (.getReplyCode client)]
      (when-not (FTPReply/isPositiveCompletion reply)
        (.disconnect client)
        (throw (ex-info (format "FTP Connection refused with reply code `%s'." reply)
                        {:reply-code reply
                         :reply-string (.getReplyString client)}))))

    ;; configurations after the connect.
    (when (false? (.login client user password))
      (throw (ex-info "FTP login failed." {:user user :host host :port port})))
    (doto client
      ;; Sets the timeout in milliseconds to use when reading from the
      ;; data connection.
      (.setDataTimeout data-timeout-ms)
      ;; Set the time to wait between sending control connection
      ;; keepalive messages when processing file upload or download.
      (.setControlKeepAliveTimeout control-keep-alive-timeout-sec)
      ;; Set how long to wait for control keep-alive message replies.
      (.setControlKeepAliveReplyTimeout control-keep-alive-reply-timeout-ms)
      (.setFileType ({:ascii  FTP/ASCII_FILE_TYPE
                      :binary FTP/BINARY_FILE_TYPE}
                     file-type))
      #_(.setPassiveNatWorkaround enable-passive-nat-workaround?)
      #_(.setRemoteVerificationEnabled enabled-remote-verification?)
      )
    (when-not (nil? list-hidden-files?)
      (.setListHiddenFiles ^FTPClient client list-hidden-files?))
    (if (= mode :passive)
      (.enterLocalPassiveMode client)
      (.enterLocalActiveMode client))
    client))

(comment
  ;; how to accept both url and host.
  (try
    (java.net.URL. host-or-url)
    (catch java.net.MalformedURLException e
      host-or-url))
  )

(comment
  ;; not supported
  setActivePortRange
  setAutodetectUTF8
  setBufferSize
  setCopyStreamListener
  setFileStructure #{FTP/FILE_STRUCTURE FTP/RECORD_STRUCTURE FTP/PAGE_STRUCTURE}
  setFileTransferMode #{FTP/STREAM_TRANSFER_MODE FTP/BLOCK_TRANSFER_MODE FTP/COMPRESSED_TRANSFER_MODE}
  setModificationTime
  setParserFactory
  setReceieveDataSocketBufferSize
  setRemoteVerificationEnabled
  setRestartOffset
  setSendDataSocketBufferSize
  setUseEPSVwithIPv4

  setActiveExternalIPAddress
  setPassiveLocalIPAddress
  setPassiveLocalIPAddress
  setReportActiveExternalIPAddress

  ;; supported
  setDataTimeout
  setControlKeepAliveTimeout
  setControlKeepAliveReplyTimeout
  setFileType
  setListHiddenFiles

  ;; pending
  setPassiveNatWorkaround
  )


(defn ^InputStream retrieve-file-stream
  "Returns an InputStream associated with a remote file.
  To finalize the file transfer you must call completePendingCommand
  and check its return value to verify success. If this is not done,
  subsequent commands may behave unexpectedly."
  ([^FTPClient client ^String remote-file-name]
     (.retrieveFileStream ^FTPClient client ^String remote-file-name)))

(defn ^OutputStream store-file-stream
  "Returns an OutputStream associated with a remote file name.
  To finalize the file transfer you must call .completePendingCommand
  and check its return value to verify success. If this is not done,
  subsequent commands may behave unexpectedly."
  ([^FTPClient client ^String remote-file-name]
     (.storeFileStream ^FTPClient client ^String remote-file-name)))


(defmacro with-retrieve-file-stream
  "Opens input-stream associated with remote-file-name to read,
  executes body to read, closes the stream, and invoke
  `.completePendingCommand` to complete a FTP transaction."
  [[^InputStream input-stream ^String remote-file-name ^FTPClient client] & body]
  {:pre [input-stream remote-file-name client]}
  `(try
     (with-open [~input-stream (.retrieveFileStream ~client ~remote-file-name)]
       ~@body)
     (finally
       (when-not (.completePendingCommand ~client)
         (throw (ex-info ".retrieveFileStream incomplete"
                         {:remote-file-name ~remote-file-name
                          :client ~client}))))))

(defmacro with-store-file-stream
  "Opens output-stream associated with remote-file-name to write,
  executes body to write, closes the stream, and invoke
  `.completePendingCommand` to complete a FTP transaction."
  [[^OutputStream output-stream ^String remote-file-name ^FTPClient client] & body]
  {:pre [output-stream remote-file-name client]}
  `(try
     (with-open [~output-stream (.storeFileStream ~client ~remote-file-name)]
       ~@body)
     (finally
       (when-not (.completePendingCommand ~client)
         (throw (ex-info ".storeFileStream incomplete"
                         {:remote-file-name ~remote-file-name
                          :client ~client}))))))

(defmacro with-append-file-stream
  "Opens output-stream associated with remote-file-name to append its content,
  executes body to append, closes the stream, and invoke
  `.completePendingCommand` to complete a FTP transaction."
  [[^OutputStream output-stream ^String remote-file-name ^FTPClient client] & body]
  {:pre [output-stream remote-file-name client]}
  `(try
     (with-open [~output-stream (.appendFileStream ~client ~remote-file-name)]
       ~@body)
     (finally
       (when-not (.completePendingCommand ~client)
         (throw (ex-info ".appendFileStream incomplete"
                         {:remote-file-name ~remote-file-name
                          :client ~client}))))))


(defn get
  "Gets a remote file and write to local file-system."
  ([^FTPClient client ^String remote-file]
   (get client remote-file (-> remote-file io/as-file .getName)))
  ([^FTPClient client ^String remote-file local-file]
   (with-open [os (io/output-stream (io/as-file local-file))]
     (.retrieveFile ^FTPClient client ^String remote-file ^OutputStream os))))

(defn put
  "Puts a local file to remote."
  ([^FTPClient client local-file]
   (put client local-file (-> local-file io/as-file .getName)))
  ([^FTPClient client local-file ^String remote-file]
   (with-open [is (io/input-stream (io/as-file local-file))]
     (.storeFile ^FTPClient client ^String remote-file ^InputStream is))))

(defn ls
  "Lists files in current directory."
  [^FTPClient client]
  (seq (.listFiles ^FTPClient client)))

(defn cd
  "Changes directory."
  [^FTPClient client dir]
  (.changeWorkingDirectory ^FTPClient client ^String dir))

(defmacro with-rewinding-directory
  "Rewinds current directory when exit body."
  [^FTPClient client & body]
  `(let [cwd# (pwd ~client)]
     (try
       ~@body
       (finally
         (cd ~client cwd#)))))

(defn pwd
  "Prints the remote working directory."
  [^FTPClient client]
  (.printWorkingDirectory ^FTPClient client))

(defn mkdir
  "Makes a remote directory."
    [^FTPClient client ^String subdir]
  (.makeDirectory ^FTPClient client ^String subdir))

(defn mkdirs
  "Makes remote directories recursively."
  [^FTPClient client ^String subdir]
  (with-rewinding-directory client
    (doseq [dir (str/split subdir #"/")]
      (mkdir client dir)
      (cd client dir))))

(defn rm
  "Removes a remote file."
  [^FTPClient client ^String remote-file-name]
  (.deleteFile ^FTPClient client ^String remote-file-name))

(defn rmdir
  "Removes a remote empty directory."
  [^FTPClient client ^String remote-directory-name]
  (.removeDirectory ^FTPClient client ^String remote-directory-name))

(defn mv
  "Renames a remote file."
  [^FTPClient client ^String from ^String to]
  (.rename ^FTPClient client ^String from ^String to))
