(ns clj-ftp-client.with-default-connection
  "Similar to clj-ftp-client.core, but has default connection hander."
  (:refer-clojure :exclude [get])
  (:require [clj-ftp-client.core :as core])
  (:import (org.apache.commons.net.ftp FTPClient)))

(def ^:dynamic *client*
  "the default ftp client handler"
  nil)

(defn set-client! [client-OR-client-spec-map]
  (let [new-client (if (map? client-OR-client-spec-map)
                     (core/open client-OR-client-spec-map)
                     client-OR-client-spec-map)]
    (alter-var-root #'*client*
                    (constantly new-client)
                    (when (thread-bound? #'*client*)
                      (set! *client* new-client)))))

(defmacro with-client
  "Establish an FTP connection, bound to *client*, and execute the body.
  Closes connection at the end of `body`.

  client-spec-map is equivalent to `clj-ftp-client.core/open`'s argument."
  [client-spec-map & body]
  `(with-open [client# ^FTPClient (core/open ~client-spec-map)]
     (binding [*client* client#]
       ~@body)))


(defmacro ^:private extend-function [name]
  (let [core-sym (symbol "clj-ftp-client.core" (str name))
        {:keys [doc arglists]} (meta (resolve core-sym))]
    `(defn ~(with-meta name (assoc (meta name) :doc doc))
       ~@(map (fn [args]
                `(~args (~core-sym ~@args)))
              arglists)
       ~@(map (fn [args]
                `(~args (~core-sym *client* ~@args)))
              (map (comp vec rest) arglists)))))

(extend-function ls)
(extend-function cd)
(extend-function pwd)
(extend-function mkdir)
(extend-function mkdirs)
(extend-function rm)
(extend-function rmdir)
(extend-function mv)

(defn ^{:doc (:doc (meta #'core/get))}
  get
  ([^String remote-file-name]
   (core/get *client* remote-file-name))
  ([^String remote-file-name local-file]
   (core/get *client* remote-file-name local-file)))

(defn ^{:doc (:doc (meta #'core/put))}
  put
  ([client local-file]
   (core/put *client* local-file))
  ([client local-file ^String remote-file-name]
   (core/put *client* local-file remote-file-name)))


(defmacro ^{:doc (:doc (meta #'core/with-rewinding-directory))}
  with-rewinding-directory
  [[^FTPClient client] & body]
  (if client
    `(core/with-rewinding-directory client
       ~@body)
    `(core/with-rewinding-directory *client*
       ~@body)))

(alter-meta! #'with-rewinding-directory
             #(assoc % :arglists
                     '([[client] & body] [[] & body])))


(defmacro ^{:doc (:doc (meta #'core/with-retrieve-file-stream))}
  with-retrieve-file-stream
  ([[^InputStream input-stream ^String remote-file-name ^FTPClient client] & body]
   (if client
     `(core/with-retrieve-file-stream [^InputStream ~input-stream ^String ~remote-file-name ^FTPClient ~client]
        ~@body)
     `(core/with-retrieve-file-stream [^InputStream ~input-stream ^String ~remote-file-name ^FTPClient *client*]
        ~@body))))

(alter-meta! #'with-retrieve-file-stream
             #(assoc % :arglists
                     '([[input-stream remote-file-name client] & body]
                       [[input-stream remote-file-name] & body])))


(defmacro ^{:doc (:doc (meta #'core/with-store-file-stream))}
  with-store-file-stream
  [[^OutputStream output-stream ^String remote-file-name ^FTPClient client] & body]
  (if client
    `(core/with-store-file-stream [^OutputStream ~output-stream ^String ~remote-file-name ^FTPClient ~client]
       ~@body)
    `(core/with-store-file-stream [^OutputStream ~output-stream ^String ~remote-file-name ^FTPClient *client*]
       ~@body)))

(alter-meta! #'with-store-file-stream
             #(assoc % :arglists
                     '([[output-stream remote-file-name client] & body]
                       [[output-stream remote-file-name] & body])))


(defmacro ^{:doc (:doc (meta #'core/with-append-file-stream))}
  with-append-file-stream
  [[^OutputStream output-stream ^String remote-file-name ^FTPClient client] & body]
  (if client
    `(core/with-append-file-stream [^OutputStream ~output-stream ^String ~remote-file-name ^FTPClient ~client]
       ~@body)
    `(core/with-append-file-stream [^OutputStream ~output-stream ^String ~remote-file-name ^FTPClient *client*]
       ~@body)))

(alter-meta! #'with-append-file-stream
             #(assoc % :arglists
                     '([[output-stream remote-file-name client] & body]
                       [[output-stream remote-file-name] & body])))
