(ns clojuress.rlang
  (:require [clojure.string :as string]
            [clojuress.protocols :as prot]
            [clojuress.util :refer [starts-with?]]
            [tech.resource :as resource]
            [clojuress.robject :refer [->RObject]])
  (:import clojuress.robject.RObject))

(defn- rand-name []
  (-> (java.util.UUID/randomUUID)
      (string/replace "-" "_")
      (->> (str "x"))))

(defn- object-name->memory-place [obj-name]
  (format ".memory$%s" obj-name))

(defn- code-that-remembers [obj-name code]
  (format "%s <- {%s}; 'ok'"
          (object-name->memory-place obj-name)
          code))

(defn- code-to-forget [obj-name]
  (format "%s <- NULL; 'ok'"
          (object-name->memory-place obj-name)))

(defn init-session-memory [session]
  (prot/eval-r->java session ".memory <- list()"))

(defn init-session [session]
  (init-session-memory session)
  session)

(defn forget [obj-name session]
  (let [returned (->> obj-name
                      code-to-forget
                      (prot/eval-r->java session))]
    (assert (->> returned
                (prot/java->clj session)
                (= ["ok"])))))

(defn eval-r [code session]
  (let [obj-name (rand-name)
        returned    (->> code
                         (code-that-remembers obj-name)
                         (prot/eval-r->java session))]
    (assert (->> returned
                 (prot/java->clj session)
                 (= ["ok"])))
    (-> (->RObject obj-name session code)
        (resource/track
         #(do (println [:releasing obj-name])
              (forget obj-name session))
         :gc))))

(defn java->r-specified-type [java-object type session]
  (prot/java->specified-type session java-object type))

(defn r-function-on-obj [{:keys [session] :as r-object}
                         function-name return-type]
  (->> r-object
       :object-name
       object-name->memory-place
       (format "%s(%s)" function-name)
       (prot/eval-r->java session)
       (#(prot/java->specified-type session % return-type))))

(defn r-class [r-object]
  (vec
   (r-function-on-obj
    r-object "class" :strings)))

(defn names [r-object]
  (vec
   (r-function-on-obj
    r-object "names" :strings)))

(defn shape [r-object]
  (vec
   (r-function-on-obj
    r-object "dim" :ints)))

(defn r->java [{:keys [session] :as r-object}]
  (->> r-object
       :object-name
       object-name->memory-place
       (prot/get-r->java session)))

(defn java->r [java-object session]
  (if (instance? RObject java-object)
    java-object
    (let [obj-name (rand-name)]
      (prot/java->r-set session
                        (object-name->memory-place
                         obj-name)
                        java-object)
      (->RObject obj-name session nil))))

(defn apply-function [r-function
                      r-args
                      session]
  (let [code
        (format
         "%s(%s)"
         (-> r-function
             :object-name
             object-name->memory-place)
         (->> r-args
              (map (fn [arg]
                     (let [[arg-name arg-value]
                           (if (starts-with? arg :=)
                             (rest arg)
                             [nil arg])]
                       (str (when arg-name
                              (str (name arg-name) "="))
                            (-> arg-value
                                (->> (prot/clj->java session))
                                (java->r session)
                                :object-name
                                object-name->memory-place)))))
              (string/join ", ")))]
    (eval-r code session)))
