(ns gorilla-notes.static-rendering
  (:require [clojure.java.io :as io]
            [clojure.string :as string]
            [gorilla-notes.state :as state]
            [selmer.parser]
            [babashka.fs :as fs]
            [scicloj.tempfiles.api :as tempfiles]))

;; https://stackoverflow.com/a/19297746
(defn copy [uri file]
  (io/make-parents file)
  (with-open [in  (io/input-stream uri)
              out (io/output-stream file)]
    (io/copy in out)))

(defn static-html [state-str]
  (-> "public/gorilla-notes/index.html"
      io/resource
      slurp
      (selmer.parser/render
       (merge
        (state/page-options)
        {:state-override (format "<script id=\"state\" type=\"text\">%s</script>"
                                 (pr-str state-str))}))))

(defn render-state! [state html-path]
  (io/make-parents html-path)
  (-> state
      (select-keys [:options :ids :id->content])
      (assoc-in [:options :auto-scroll?] false)
      pr-str
      static-html
      (->> (spit html-path)))
  (let [html-parent-path (-> html-path
                             fs/parent
                             str)
        current-tempdir (tempfiles/session-tempdir!)
        target-tempdir (str html-parent-path
                            (string/replace current-tempdir #"/tmp" ""))]
    (fs/create-dirs target-tempdir)
    (fs/copy-tree current-tempdir
                  target-tempdir)
    (copy (io/resource "public/gorilla-notes/js/compiled/main.js")
          (str html-parent-path
               "/gorilla-notes/js/compiled/main.js"))))

(defn render-current-state! [html-path]
  (render-state! @state/*state html-path))


