(ns notespace.markdown
  (:require [clojure.string :as string]
            [clojure.tools.reader :as tr]
            [clojure.tools.reader.reader-types :as rts]))

(set! *print-length* 1000)

(defonce example-doc
  (slurp "https://raw.githubusercontent.com/scicloj/tablecloth/master/docs/index.Rmd"))

(defn chunk-end? [line]
  (= line "```"))

(defn clojure-chunk-beginning? [line]
  (re-matches #"```\{clojure.*\}" line))

(defn clojure-chunk-options [line]
  (-> line
      (string/replace #"^```\{clojure " "")
      (string/replace #"\}$" "")))

(defn markdown->collected-chunks [markdown]
  (loop [lines                 (string/split markdown #"\n")
         current-clojure-chunk-code    nil
         current-clojure-chunk-options nil
         result                []]
    (if-let [current-line (first lines)]
      (if current-clojure-chunk-code
        (if (chunk-end? current-line)
          (recur (rest lines)
                 nil
                 nil
                 (conj result {:clojure-chunk? true
                               :code current-clojure-chunk-code
                               :options current-clojure-chunk-options}))
          (recur (rest lines)
                 (conj current-clojure-chunk-code current-line)
                 current-clojure-chunk-options
                 result))
        (if (clojure-chunk-beginning? current-line)
          (recur (rest lines)
                 []
                 (clojure-chunk-options current-line)
                 result)
          (recur (rest lines)
                 nil
                 nil
                 (conj result current-line))))
      result)))

(defn clojure-chunk->markdown [{:keys [clojure-chunk? code options]}]
  (when clojure-chunk?
    (str "```{clojure "
         options
         "}\n"
         (string/join "\n" code)
         "\n```")))

(defn collected-chunks->markdown [mwecc]
  (->> mwecc
       (map #(or (clojure-chunk->markdown %)
                 %))
       (string/join "\n")
       (format "%s\n")))

(->> example-doc
     (spit "/tmp/0.Rmd"))

(->> example-doc
     markdown->collected-chunks
     collected-chunks->markdown
     (spit "/tmp/a.Rmd"))

(->> example-doc
     markdown->collected-chunks
     collected-chunks->markdown
     (= example-doc))

(->> example-doc
     markdown->collected-chunks
     (partition-by string?))

(defn inner-pr-str [s]
  (let [s1 (pr-str s)
        n (count s1)]
    (subs s1 1 (dec n))))

(defn collected-chunks->clojure [mwecc]
  (->> mwecc
       (partition-by string?)
       (mapcat
        (fn [part]
          (cond
            ;;
            (-> part first string?)
            [(->> part
                  (map inner-pr-str)
                  (string/join "\n")
                  (format "[\"%s\"]"))]
            ;;
            (-> part first :clojure-chunk?)
            [(->> part
                  (mapcat
                   (fn [{:keys [code options]}]
                     (cons (format "^{:chunk-options %s}[]\n"
                                   (pr-str options))
                           code)))
                  (string/join "\n"))])))
       (string/join "\n\n\n")
       (format "(ns index)\n%s")))


(->> example-doc
     markdown->collected-chunks)

(->> example-doc
     markdown->collected-chunks
     collected-chunks->clojure
     (spit "/workspace/clones/tablecloth/docs/index.clj"))


(defn clojure->forms [clojure]
  (->> clojure
       rts/source-logging-push-back-reader
       repeat
       (map #(tr/read % false :EOF))
       (take-while (partial not= :EOF))))

(defn vector-of-strings? [form]
  (true? (and (vector? form)
              (seq form)
              (every? string? form))))

(defn ->chunk [chunk-options forms]
  {:clojure-chunk? true
   :code (->> forms
              (filter #(-> % meta :chunk-options not))
              (map #(-> % meta :source)))
   :options chunk-options})

(defn clojure->collected-chunks [clojure]
  (->> clojure
       clojure->forms
       rest ; removing the first ns form
       (partition-by vector-of-strings?)
       (mapcat (fn [forms]
                 (if (-> forms first vector-of-strings?)
                   (apply concat forms)
                   (let [{:keys [chunk-options]} (-> forms first meta)]
                     [(->chunk chunk-options
                               (filter #(not= % [])
                                       forms))]))))))

(->> "/workspace/clones/tablecloth/docs/index.clj"
     slurp
     clojure->collected-chunks
     collected-chunks->markdown
     (spit "/tmp/index.Rmd"))

(clojure.java.shell/sh
 "meld"
 "/tmp/index.Rmd"
 "/workspace/clones/tablecloth/docs/index.Rmd")
