(ns notespace.markdown
  (:require [clojure.string :as string]
            [clojure.tools.reader :as tr]
            [clojure.tools.reader.reader-types :as rts]))

(defn dbg [msg data]
  (println (pr-str [msg data]))
  data)

(defn chunk-end? [line]
  (= line "```"))

(defn clojure-chunk-beginning? [line]
  (or (re-matches #"```\{clojure.*\}" line)
      (re-matches #"```clojure.*" line)))

(defn clojure-chunk-options [line]
  (-> line
      (string/replace #"^```\{clojure" "")
      (string/replace #"\}$" "")))

(defn markdown->collected-chunks [markdown]
  (loop [lines                 (string/split markdown #"\n")
         current-clojure-chunk-code    nil
         current-clojure-chunk-options nil
         result                []]
    (if-let [current-line (first lines)]
      (if current-clojure-chunk-code
        (if (chunk-end? current-line)
          (recur (rest lines)
                 nil
                 nil
                 (conj result {:clojure-chunk? true
                               :code current-clojure-chunk-code
                               :options current-clojure-chunk-options}))
          (recur (rest lines)
                 (conj current-clojure-chunk-code current-line)
                 current-clojure-chunk-options
                 result))
        (if (clojure-chunk-beginning? current-line)
          (recur (rest lines)
                 []
                 (clojure-chunk-options current-line)
                 result)
          (recur (rest lines)
                 nil
                 nil
                 (conj result current-line))))
      result)))

(defn clojure-chunk->markdown [{:keys [clojure-chunk? code options]}]
  (when clojure-chunk?
    (str "```{clojure"
         options
         "}\n"
         (string/join "\n" code)
         "\n```")))

(defn collected-chunks->markdown [collected-chunks]
  (->> collected-chunks
       (map (fn [text-or-chunk]
              ;; (when (clojure-chunk->markdown text-or-chunk)
              ;;   (println [text-or-chunk
              ;;             (-> text-or-chunk
              ;;                 clojure-chunk->markdown
              ;;                 (or text-or-chunk))]))
              (-> text-or-chunk
                  clojure-chunk->markdown
                  (or text-or-chunk))))
       (string/join "\n")
       (format "%s\n")))



(defn clojure-chunk->clojure [clojure-chunk]
  (->> clojure-chunk
       (mapcat
        (fn [{:keys [code options]}]
          (cons (format "^{:chunk-options %s}[]\n"
                        (pr-str options))
                code)))
       (string/join "\n")))

(defn inner-pr-str
  "This acts like pr-str, but removes the first and last characters
  (which are the quotes, in the case of a string input)."
  [s]
  (let [s1 (pr-str s)
        n (count s1)]
    (subs s1 1 (dec n))))

(defn text-lines->clojure [text-lines]
  (->> text-lines
       (map inner-pr-str)
       (string/join "\n")
       (format "[\"%s\"]")))

(defn collected-chunks->clojure [mwecc]
  (->> mwecc
       (partition-by string?)
       (mapcat
        (fn [part]
          (cond
            ;;
            (-> part first string?)
            [(->> part
                  text-lines->clojure)]
            ;;
            (-> part first :clojure-chunk?)
            [(->> part
                  clojure-chunk->clojure)])))
       (string/join "\n\n")
       (format "(ns index)\n%s")))

(defn clojure->forms [clojure]
  (->> clojure
       rts/source-logging-push-back-reader
       repeat
       (map #(tr/read % false :EOF))
       (take-while (partial not= :EOF))))

(defn vector-of-strings? [form]
  (true? (and (vector? form)
              (seq form)
              (every? string? form))))

(defn options-line? [line]
  (re-matches #";; chunk-options:" line))

(defn ->chunk [chunk-options forms]
  {:clojure-chunk? true
   :code (->> forms
              (filter #(-> % meta :chunk-options not))
              (map #(-> % meta :source)))
   :options chunk-options})

(defn clojure->collected-chunks [clojure]
  (->> clojure
       clojure->forms
       rest ; removing the first ns form
       (partition-by vector-of-strings?)
       (mapcat (fn [forms]
                 (println (pr-str [:forms forms]))
                 (if (-> forms first vector-of-strings?)
                   (apply concat forms)
                   (let [{:keys [chunk-options]} (-> forms first meta)]
                     [(->chunk chunk-options
                               (filter #(not= % [])
                                       forms))]))))))

(comment
  (->> "/workspace/Dropbox/projects/scicloj/sicmutils-notespace-example/exercise-1.21_-a-dumbbell.nextjournal.md"
       slurp
       markdown->collected-chunks
       collected-chunks->clojure
       (spit "/tmp/index.clj")))

;; (->> "/workspace/clones/tablecloth/docs/index.clj"
;;      slurp
;;      clojure->collected-chunks
;;      collected-chunks->markdown
;;      (spit "/tmp/index.Rmd"))

;; (clojure.java.shell/sh
;;  "meld"
;;  "/tmp/index.Rmd"
;;  "/workspace/clones/tablecloth/docs/index.Rmd")
