(ns scicloj.notespace.v4.loop)

; a note is a map with:
;; [source   ; the source code
;;  location ; a map with line,row of beginning,end
;;  kind     ; ...
;;  value    ; if exists, evaluation value if exists
;;  realized-value ; if exists, hdereffed value when value is dereffable
;;  status
;;  ;; for regular value: :not-evaluating-yet / :evaluating-now / :eval-failed / :eval-succeeded
;;  ;; for realized value: :irrelevant / :not-evaluating-yet / :evaluating-now / :eval-failed / :eval-succeeded
;;  ]

(defonce initial-state
  {:ns->notes    {;ns1 [note1 note2]
                  } ; for every namespace, a vector of notes
   })

(defonce *state (atom initial-state))

;; Problems:
;; - how to know that an nREPL request was sent?
;; - reading notes correctly (after file-save, nREPL load-file): comments, non-IOBj, ... (probably neglecting some cases such as #"..." regexp, etc.)
;; - diffing two lists of notes, to see which are already familiar to us -- a diff algorithm could be one way to decide (maybe using patch -- see below)
;; - comparing an evaluated region, to see if it seems compatible with last file-save (or nREPL load-file) -- an open question? using diff somehow?
;; - can we mirror the editor buffer without saving (maybe use some temporary file)?
;; - UX problems:
;;   - how to interact with delays?
;;   - how to eval the whole ns and inform Notespace? (load-file doesn't return enough information, so maybe we need an editor binding to mark the whole buffer and then eval it?) -- "send to notespace"
;;   - how to communicate the fact that a newly evaluated region is not known to us?

;; notes0
;; [a b c d]

;; notes1
;; [a b e f d g]

;; (diff notes0 notes1) 
;; => delta:
;; -c +e +f +g (with positions)

;; (patch notes0 delta)
;; => notes1

;; (patch (repeat 4 nil) delta)
;; => [nil nil e f nil g]


;; ;; https://newbedev.com/clojure-remove-item-from-vector-at-a-specified-location
;; (defn vec-remove
;;   [v pos]
;;   (vec (concat (subvec v 0 pos)
;;                (subvec v (inc pos)))))

;; (defn vec-add
;;   [v pos elem]
;;   (vec (concat (subvec v 0 pos)
;;                [elem]
;;                (subvec v pos))))

;; (defn vec-apply-edit [v edit]
;;   (->> (let [[[position] op & values] edit
;;              value (first values)]
;;          (case op
;;            :- (vec-remove v
;;                           position)
;;            :+ (vec-add v
;;                        position
;;                        value)
;;            :r (if position
;;                 (assoc v
;;                        position
;;                        value)
;;                 value)))))

;; (fact (-> [1 2 3]
;;           (v4.editscript/vec-remove 1))
;;       => [1 3])

;; (fact (-> [1 2 3]
;;           (v4.editscript/vec-add 1 4))
;;       => [1 4 2 3])

;; (fact (-> [1 2 3]
;;           (v4.editscript/vec-apply-edit [[1] :-]))
;;       => [1 3])

;; (fact (-> [1 2 3]
;;           (v4.editscript/apply-edits [[[1] :+ 4]]))
;;       => [1 4 2 3])

;; (fact (-> [1 2 3]
;;           (v4.editscript/apply-edits [[[1] :r 4]]))
;;       => [1 4 3])

;; (fact (-> [1 2 3]
;;           (v4.editscript/apply-edits [[[] :r [4]]]))
;;       => [4])

;; (defn update-view [last-value notes]
;;   (->> [:div
;;         (title "last value")
;;         (or (some-> last-value v4.render/render)
;;             "-")
;;         [:hr]
;;         (title "notes")
;;         (notes->hiccup notes)]
;;        (gn/assoc-note! 0)))


(defn apply-note-edit-to-frontend! [[[position & _] op & _]]
  (case op
    :- (v4.frontend/remove-widget! position)
    :+ (v4.frontend/add-widget! position
                                nil)
    :r (when position
         (v4.frontend/assoc-widget! position
                                    nil))))

(defn apply-note-edits-to-frontend! [path key->edits]
  (some->> key->edits
           :path->notes
           path
           (run! apply-note-edit-to-frontend!)))

(defn apply-edits-to-frontend! [state key->edits]
  (reset-frontend-header! state)
  (apply-note-edits-to-frontend! [(:current-path state)
                                  key->edits]))

#_(if path-change?
    (reset-frontend! @*state)
    ;; else
    (apply-edits-to-frontend! key->edits
                              (:current-path @*state)))


path-change? (some-> key->edits
                     :current-path
                     first
                     last
                     (not= (:current-path @*state)))


(defn add-status [message]
  [:messages [[[0 :+ message]]]])

(defn set-current-path [path]
  {:current-path [[[] :r path]]})

(defn set-request-path [request-id path]
  {:request-id->path [[[request-id] :r path]]})

(defn set-last-value [value]
  {:last-value [[[] :r value]]})

(defn edit-notes [path edits]
  {:path->notes
   (->> edits
        (map (fn [edit]
               (-> edit
                   (update 0 #(vec (cons path %)))))))})

;; (defn log-data-and-broadcast
;;   ([data]
;;    (log-data-and-broadcast nil data))
;;   ([title data]
;;    (log-data title data)
;;    (gn/add-note!
;;     [:div [:p title]
;;      [:p/code (-> data pp/pprint with-out-str)]])
;;    data))

;; (comment
;;   (log-data-and-broadcast ".")
;;   (gn/reset-notes!))



;; (let [path-change? (some-> key->edits
;;                            :current-path
;;                            first
;;                            last
;;                            (not= (:current-path @*state)))]
;;   (apply-edits-to-state!)
;;   (if path-change?
;;       (reset-frontend! @*state)
;;       ;; else
;;       (apply-edits-to-frontend! key->edits
;;                                 (:current-path @*state))))
 

(defn apply-note-edit-to-frontend! [[[position & _] op & args]]
  (case op
    :- (v4.frontend/remove-widget! position)
    :+ (v4.frontend/add-widget! position
                                (-> args
                                    first
                                    v4.view/note->hiccup))
    :r (if position
         (v4.frontend/assoc-widget! position
                                    (-> args
                                        first
                                        v4.view/note->hiccup))
         ;; else
         (do (v4.frontend/reset-widgets!)
             (->> args
                  first
                  (map v4.view/note->hiccup)
                  (run! v4.frontend/add-widget!))))))




(ns scicloj.notespace.v4.render
  (:require [scicloj.notespace.v4.image :as v4.image]))


(defprotocol Renderable
  :extend-via-metadata true
  (render [this]))

(extend-protocol Renderable
  Object
  (render [o]
    [:p/code
     (pr-str o)])
  nil
  (render [this]
    [:p/code "nil"])
  java.awt.image.BufferedImage
  (render [this]
    (v4.image/buffered-image->hiccup this)))

(defn as-hiccup [content]
  (vary-meta
   content
   assoc 'scicloj.notespace.v4.render/render identity))

(defn as-vega [content]
  (vary-meta
   content
   assoc 'scicloj.notespace.v4.render/render
   (fn [spec]
     [:p/vega spec])))
