(ns sgr-db.record
  (:require [monger.core       :as        mg]
            [monger.collection :as        mc]
            [monger.util       :as      util]
            [monger.result     :refer  [ok?]]
            [monger.joda-time  :as joda-time]
            [monger.util         :refer :all]
            [clj-time.core     :as      time]
            [schema.core       :as         s]

            [sgr-db.config     :refer :all]
            [sgr-db.models     :as    models]
            [sgr-db.gridfs     :refer   :all]))

(defn- insert-record
  "Place a new record in the database. Must be a valid record, returns true if successfully inserted, otherwise false."
  [dbname coll record]
  (let [db (mg/get-db conn dbname)]
    (try
      (ok? (mc/insert db coll (s/validate models/Record record)))
      (catch Exception e
        (do (println (str "Exception SGR-db.record.insert-record: " (.getMessage e)))
          false)))))

(defn- update-record
  "example:
  (update-record 'mongo-test' 'records' {:record-id 'd2708627-1f44-4b12-9bd6-4df906cc6a37'} {'$set' {:gender 'male'}})"
  [dbname coll conditions document]
  (let [db (mg/get-db conn dbname)]
    (ok? (mc/update db coll conditions document))))

(defn get-records
  ([dbname coll]
   (let [db (mg/get-db conn dbname)]
     (if-let [records (mc/find db coll)]
       records)))
  ([dbname coll query]
   (let [db (mg/get-db conn dbname)]
     (if-let [records (mc/find db coll query)]
       records))))

(defn get-records-maps
  ([dbname coll]
   (let [db (mg/get-db conn dbname)]
     (if-let [records (mc/find-maps db coll)]
       records)))
  ([dbname coll query]
   (let [db (mg/get-db conn dbname)]
     (if-let [records (mc/find-maps db coll query)]
       records))))

(defn get-record
  [dbname coll query]
  (let [db (mg/get-db conn dbname)]
    (if-let [record (mc/find-one db coll query)]
      record)))

(defn get-record-as-map
  [dbname coll query]
  (let [db (mg/get-db conn dbname)]
    (if-let [record (mc/find-one-as-map db
                                      coll
                                      query)]
      record)))

(defn set-mfcc
  {:doc "<conditions or record>"}
  [dbname coll conditions mfcc]
  (update-record dbname coll conditions {"$set" {:mfcc mfcc}}))

(defn get-mfcc
  {:doc "<condition or record>"}
  [dbname coll conditions]
  (:mfcc (get-record-as-map dbname coll conditions)))

(defn get-all-mfcc
  [dbname coll]
  (keep #(if (:mfcc %) %) (get-records-maps dbname coll)))

(defn create-record
  {:doc
   "{:path path-to-record
     :converted?  true||false
     :mfcc        mfcc||false
    }"}
  [dbname coll record]
  (try
    (let [gripid (gridfs-store (:path record) dbname)
          r     {:record-id  gripid
                 :converted? (:converted? record)
                 :revisions  {:revision-id gripid
                              :grid-id     gripid}
                 :mfcc (if (:mfcc record) (:mfcc record) [])
                 :path       (:path record)}]
      (if (insert-record dbname coll r)
        ;;(if-let [mfcc (:mfcc record)]
        ;;  (do
         ;;   (println mfcc)
         ;;   (set-mfcc dbname coll r mfcc))
          r;)
        false))
    (catch Exception e
      (do (println (str "Exception SGR-db.record.create-record: " (.getMessage e)))
        false))))
