(ns shepherd.password
  (:require [crypto.password.bcrypt]
            [crypto.password.scrypt]
            [crypto.password.pbkdf2]))


(def bcrypt
  "Encrypts a string with BCrypt.

   Optional Parameters:
     work-factor - log2 of the number of hashing rounds,
                   defaults to 11

   Variations:
     (bcyrpt raw)
     (bcrypt raw work-factor)"

  crypto.password.bcrypt/encrypt)


(def scrypt
  "Encrypts a string with SCrypt.

   Optional Parameters:
     n - the CPU cost, must be a power of 2, defaults to 2^15
     r - the memory cost, defaults to 8
     p - the parallelization parameter, defaults to 1

   Variations:
     (scrypt raw)
     (scrypt raw n)
     (scrypt raw n r p)"

  crypto.password.scrypt/encrypt)


(def pbkdf2
  "Encrypts a string with PBKDF2.

   Encodes the number of iterations and salt in following format:
     <iterations>$<salt>$<encrypted string>

   All elements in the ouput string are Base64 encoded.

   Optional Parameters:
     iterations - defaults to 100,000
     salt - defaults to 8 randomly generated bytes

   Variations:
     (pbkdf2 raw)
     (pbkdf2 raw iterations)
     (pbkdf2 raw iterations salt)"

  crypto.password.pbkdf2/encrypt)


(def check-bcrypt
  "Compares a raw string with a BCrypt-encrypted string.

   Returns true on match.

   Variations:
     (check-bcrypt raw encrypted)"
  
  crypto.password.bcrypt/check)


(def check-scrypt
  "Compares a raw string with a SCrypt-encrypted string.

   Returns true on match.

   Variations:
     (check-scrypt raw encrypted)"
  
  crypto.password.scrypt/check)


(def check-pbkdf2
  "Compares a raw string with a PBKDF2-encrypted string.

   Returns true on match.

   Variations:
     (check-pbkdf2 raw encrypted)"

  crypto.password.pbkdf2/check)
