(ns signal.logging
  (:require [taoensso.encore :as enc] ;; encore comes via timbre
            [taoensso.timbre :as timbre]
            [timbre-json-appender.core :as tas]
            [clojure.string :as string])
  (:import [org.slf4j.bridge SLF4JBridgeHandler]))

(defn- valid-log-level-keyword? [level]
  (contains? #{:trace :debug :info :warn :error :fatal} level))

(defmacro get-namespace-prefix
  "Returns the namespace prefix for use in logging config.
   
   e.g. when called from the namespace signal.organisations-api, returns \"signal\""
  []
  `(first (string/split (str ~*ns*) #"\.")))

(defn- create-log-config
  ([namespace log-level]
   (cond-> [["taoensso.*" :error]]
     (nil? namespace) (conj ["*" (or log-level :info)])
     (some? namespace) (-> (conj [namespace (or log-level :info)])
                           (conj ["*" :warn])))))

(defn set-log-level!
  "Set the log level to 'new-value' (either a string or a keyword).
   See https://github.com/ptaoussanis/timbre for all possible values.
   Return a boolean for successful, along with a message."
  ([namespace log-level]
   (if-let [parsed-log-value (cond
                               (sequential? log-level) log-level
                               (valid-log-level-keyword? log-level) (create-log-config namespace log-level)
                               :else nil)]
     (do
       (timbre/reportf "Setting log level to %s" parsed-log-value)
       (timbre/set-level! parsed-log-value)
       [true "OK"])
     (do
       (timbre/reportf "Invalid log level: %s" log-level)
       [false (str "Invalid log level: " log-level)])))
  ([log-level]
   (set-log-level! nil log-level)))

(defn- merge-context-middleware
  "Middleware which merges the timbre context into the vargs list.
   
   Provided args take precedent over context args.
   
   This is a no-op with format style args."
  [{:keys [context msg-type vargs] :as data}]
  (if (= :p msg-type)
    ;; re-use timbre-json-appender varg parsing for consistency
    (let [{:keys [message args]} (tas/collect-vargs vargs)]
      (assoc data :vargs (conj (if message [message] []) (-> (merge context args)
                                                             seq
                                                             flatten))))
    data))

(defn output-fn
  "Modification of timbre's default output-fn which adds the thread name.
   
   See `timbre/default-output-fn`.

   Default (fn [data]) -> string output fn.
    Use`(partial output-fn <opts-map>)` to modify default opts."
  ([data] (output-fn nil data))
  ([opts data]
   (let [{:keys [no-stacktrace?]} opts
         {:keys [level ?err #_vargs msg_ ?ns-str ?file hostname_
                 timestamp_ ?line]} data]
     (str
      (force timestamp_)
      " "
      (force hostname_) " "
      (string/upper-case (name level))  " "
      (.getName (Thread/currentThread)) " "
      "[" (or ?ns-str ?file "?") ":" (or ?line "?") "] - "
      (force msg_)
      (when-not no-stacktrace?
        (when-let [err ?err]
          (str enc/system-newline (timbre/stacktrace err opts))))))))

(defn init!
  "Initialise Timbre configuration and SLF4J bridge."
  ([config]
   (SLF4JBridgeHandler/removeHandlersForRootLogger)
   (SLF4JBridgeHandler/install)

   (let [config-level (:min-level config)
         resolved-level (if (sequential? config-level)
                          config-level
                          (create-log-config (:ns config) config-level))
         json? (some? (:json config))
         pretty? (some? (:pretty (:json config)))
         json-appender (when json? (tas/json-appender {:pretty pretty?}))]

     (timbre/set-config! (merge
                          {:min-level resolved-level
                           :ns-whitelist []
                           :ns-blacklist []
                           :middleware (concat
                                        ;; only apply context middleware if not using json appender
                                        ;; timbre-json-appender handles context itself
                                        (if json? [] [merge-context-middleware])
                                        (or (:middleware config) []))
                           :appenders (merge (if json?
                                               {:json json-appender}
                                               {:println (timbre/println-appender {:stream :auto})})
                                             (:appenders config))
                           :timestamp-opts timbre/default-timestamp-opts
                           ;; output-fn is used by the default print-ln appender: :println (timbre/println-appender {:stream :auto})
                           ;; it is not used by timbre-json-appender
                           :output-fn output-fn}
                          (-> config
                              (dissoc config :appenders)
                              (dissoc config :middleware)
                              (dissoc config :json)
                              (dissoc config :level)
                              (dissoc config :min-level))))))
  ([]
   (init! {:json true})))
