(ns signal.logging.ring
  (:require [taoensso.timbre :as timbre]
            ;; jsonista is loaded via timbre-json-appender.core 
            [jsonista.core :as j]
            [signal.logging :as logging]))

(defn- log-success [level request response]
  (let [{:keys [uri query-string request-method]} request
        {status :status body :body} response]
    (timbre/log level
                :path uri
                :query-string (or query-string "")
                :method request-method
                :status status
                :bytes-sent (count body))))

(defn- log-error [error request]
  (let [{:keys [uri query-string request-method]} request]
    (timbre/log :error error "Failed to handle request"
                :path uri
                :query-string (or query-string "")
                :method request-method)))

(defn with-logging
  "Ring middleware which logs the result of every request."
  ([handler {:keys [level] :or {level :info} :as _options}]
   (fn
     ([request]
      (try
        (let [response (handler request)]
          (log-success level request response)
          response)
        (catch Throwable t
          (log-error t request)
          (throw t))))
     ([request respond raise]
      (handler request
               (fn [response]
                 (log-success level request response)
                 (respond response))
               (fn [t]
                 (log-error t request)
                 (raise t))))))
  ([handler]
   (with-logging handler nil)))

(defn set-log-level-handler
  "Ring handler which allows setting the log level at runtime. to the given 'level' parameter.
   
   Should be attached to a GET request and called with a level parameter, e.g.
   
   GET /api/loglevel?level=[new-log-level]"
  [request]
  (let [level-string (get-in request [:params "level"])
        level (try (j/read-value level-string)
                   (catch Throwable _
                     level-string))
        [success message] (logging/set-log-level! level)]
    (if success
      {:status  200
       :body     message}
      {:status  400
       :body     message})))
