# Summaries

## Intro

Summaries is a clojure/clojurescript library without external
dependencies for enriching a nested data structure with data derived
from the existing data in the structure.

## An example

You have data about a classical orchestra. There are musicians with
wood, brass and strings instruments. The following represents the data
in a clojure map.

```clojure
(def orchestra
  {:conductor "Leonard B."
   :musician
   [{:name "Morten K."
     :experience-level 7
     :instrument-class 'brass
     :instrument-type 'trumpet
     :venues ["hollywood bowl"
              "royal arena"]}
    {:name "Anders T."
     :experience-level 8
     :instrument-class 'brass
     :instrument-type 'french-horn
     :venues ["sydney opera house"
              "royal arena"]}
    {:name "Melanie C."
     :experience-level 4
     :instrument-class 'wood
     :instrument-type 'oboe
     :venues ["recife"]}
    {:name "Trine T."
     :experience-level 2
     :instrument-class 'wood
     :instrument-type 'clarinet
     :venues []}
    {:name "Kurt H."
     :experience-level 5
     :instrument-class 'strings
     :instrument-type 'violin
     :venues ["bourla"]}
    {:name "Mikkel G."
     :experience-level 9
     :instrument-class 'strings
     :instrument-type 'cello
     :venues ["aarhus musikhus"]}]})
```

You would now like to enrich this data with some fields. You need the
average experience level for musicians using wood instruments, the set
of venues that musicians have played at, and per musician the amount
of venues he or she has played. The following data structures describe
tasks that summaries can execute.


```clojure
(def avg-experience-wood
  {:operation 'avg
   :input-path [:musician* ['= :instrument-class 'wood] :experience-level]
   :insert-path [:summaries :avg-experience-wood]})

(def amount-venues-per-musician
  {:operation 'count
   :start-path [:musician*]
   :input-path [:venues*]
   :insert-path [:amount-venues]})

(def unique-venues-across-musicians
  {:operation 'unique
   :input-path [:musician* :venues*]
   :insert-path [:summaries :venues]})
```

You can now reduce to the desired result, using the apply-summary
function.

```clojure
(reduce (fn [acc task]
          (apply-summary task acc))
        orchestra
        [avg-experience-wood
         amount-venues-per-musician
         unique-venues-across-musicians])

;==>
{:conductor "Leonard B.",
 :musician
 [{:name "Morten K.",
   :experience-level 7,
   :instrument-class brass,
   :instrument-type trumpet,
   :venues ["hollywood bowl" "royal arena"],
   :amount-venues 2}
  {:name "Anders T.",
   :experience-level 8,
   :instrument-class brass,
   :instrument-type french-horn,
   :venues ["sydney opera house" "royal arena"],
   :amount-venues 2}
  {:name "Melanie C.",
   :experience-level 4,
   :instrument-class wood,
   :instrument-type oboe,
   :venues ["recife"],
   :amount-venues 1}
  {:name "Trine T.",
   :experience-level 2,
   :instrument-class wood,
   :instrument-type clarinet,
   :venues [],
   :amount-venues 0}
  {:name "Kurt H.",
   :experience-level 5,
   :instrument-class strings,
   :instrument-type violin,
   :venues ["bourla"],
   :amount-venues 1}
  {:name "Mikkel G.",
   :experience-level 9,
   :instrument-class strings,
   :instrument-type cello,
   :venues ["aarhus musikhus"],
   :amount-venues 1}],
 :summaries
 {:avg-experience-wood 3.0,
  :venues
  ["recife"
   "sydney opera house"
   "royal arena"
   "hollywood bowl"
   "aarhus musikhus"
   "bourla"]}}
```

Notice how the start-path enables summaries to inject data several
places in the original data structure in one go.

## Running tests in Clojurescript

```
clj -C:test -A:cljs-test
```

## Acknowledgements

The code as provided was produced by
- [Kurt Heyrman](https://github.com/qrthey)
- [Mikkel Gravgaard](https://github.com/grav)

Copyright [UFST](https://ufst.dk) (see [LICENSE](./LICENSE)).
