(ns leiningen.larder
  (:require [leiningen.clean :as lc]
            [leiningen.core.main :as lm]
            [leiningen.deploy :as ld]
            [cemerick.pomegranate.aether :as aether]
            [clojure.java.io :as io]
            [clojure.java.shell :refer [sh]]
            [clojure.string :as s]))

(defn git-version []
  (format "g%s"
          (-> (:out (sh "git" "rev-parse" "HEAD"))
              s/trim)))

(defn unstaged-changes? []
  (or (pos? (:exit (sh "git" "diff" "--quiet" "--exit-code")))
      (pos? (:exit (sh "git" "diff" "--cached" "--quiet" "--exit-code")))))

(defn uberjar-name [project]
  (format "%s-larder-release.jar" (:name project)))

(defn clean! [project]
  (assert (not (unstaged-changes?)) "You have unstaged changes - please commit or stash them before releasing!")

  (lm/info "Cleaning project...")
  (lc/clean project))

(defn release-jar! [{:keys [group target-path local-repo] :as project} {:keys [repository]}]
  (clean! project)

  (let [repository (or repository "releases")
        uberjar-file-name (uberjar-name project)
        version (git-version)]

    (lm/info "Building project...")
    (lm/resolve-and-apply (assoc project
                            :uberjar-name uberjar-file-name)
                          ["build"])

    (lm/info "Deploying project...")
    (aether/deploy
     :coordinates [(symbol group (:name project)) version]
     :jar-file (io/file target-path uberjar-file-name)
     :transfer-listener :stdout
     :repository [(ld/repo-for project repository)]
     :local-repo local-repo)

    (lm/info (format "Released %s/%s (version %s)!" group (:name project) version))))

(defn larder [project command & [repository & args]]
  (let [repository (or repository "releases")]
    (case [(keyword command) (get-in project [:larder :artifact-type] :jar)]
      [:release :jar] (release-jar! project {:repository repository})

      (binding [*out* *err*]
        (println "Invalid command:" command)))))
