/*
 * Decompiled with CFR 0.152.
 */
package org.neo4j.jdbc;

import java.io.IOException;
import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLWarning;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Map;
import java.util.Properties;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.neo4j.jdbc.AbstractConnection;
import org.neo4j.jdbc.Connections;
import org.neo4j.jdbc.Driver;
import org.neo4j.jdbc.ExecutionResult;
import org.neo4j.jdbc.IteratorResultSet;
import org.neo4j.jdbc.Neo4jDatabaseMetaData;
import org.neo4j.jdbc.Neo4jPreparedStatement;
import org.neo4j.jdbc.Neo4jStatement;
import org.neo4j.jdbc.QueryExecutor;
import org.neo4j.jdbc.Version;
import org.neo4j.jdbc.rest.Resources;
import org.neo4j.jdbc.rest.TransactionalQueryExecutor;
import org.neo4j.jdbc.util.UserAgentBuilder;
import org.restlet.Client;
import org.restlet.Context;
import org.restlet.data.Protocol;

public class Neo4jConnection
extends AbstractConnection {
    protected static final Log log = LogFactory.getLog(Neo4jConnection.class);
    private String url;
    private QueryExecutor queryExecutor;
    private boolean closed = false;
    private final Properties properties = new Properties();
    private boolean debug;
    private Driver driver;
    private Version version;
    private SQLWarning sqlWarnings;
    private boolean readonly = false;
    private boolean autoCommit = true;
    private final UserAgentBuilder userAgentBuilder;

    public Neo4jConnection(Driver driver, String jdbcUrl, Properties properties) throws SQLException {
        this.driver = driver;
        this.url = jdbcUrl;
        this.properties.putAll((Map<?, ?>)properties);
        this.debug = this.hasDebug();
        this.userAgentBuilder = new UserAgentBuilder(properties);
        String connectionUrl = Neo4jConnection.extractConnectionUrl(jdbcUrl);
        this.queryExecutor = this.createExecutor(connectionUrl, this.getUser(), this.getPassword(), this.userAgentBuilder.getAgent());
        this.version = this.queryExecutor.getVersion();
        this.validateVersion();
    }

    private static String extractConnectionUrl(String jdbcUrl) {
        if (jdbcUrl == null) {
            throw new IllegalArgumentException("JDBC URL must not be null");
        }
        if (jdbcUrl.isEmpty()) {
            throw new IllegalArgumentException("JDBC URL must not be empty");
        }
        String baseUrl = jdbcUrl.substring("jdbc:neo4j".length());
        if (baseUrl.isEmpty()) {
            throw new IllegalArgumentException("JDBC URL must specify a server");
        }
        if (baseUrl.startsWith(":http")) {
            return baseUrl.substring(1);
        }
        return baseUrl;
    }

    private QueryExecutor createExecutor(String connectionUrl, String user, String password, String userAgent) throws SQLException {
        if (connectionUrl.contains("://")) {
            String remoteUrl = connectionUrl;
            if (!connectionUrl.startsWith("http://") && !connectionUrl.startsWith("https://")) {
                remoteUrl = "http" + remoteUrl;
            }
            if (log.isDebugEnabled()) {
                log.debug("Connecting to URL " + this.url);
            }
            Client client = new Client(new Context(), Arrays.asList(Protocol.valueOf(remoteUrl.split(":")[0])), this.properties.getProperty("restlet.helperclass"));
            Resources resources = new Resources(remoteUrl, client, userAgent);
            if (user != null && password != null) {
                resources.setAuth(user, password);
            }
            try {
                Resources.DiscoveryClientResource discovery = resources.getDiscoveryResource();
                if (discovery.getTransactionPath() != null) {
                    return new TransactionalQueryExecutor(resources);
                }
                throw new SQLException("Could not connect to the Neo4j Server at " + remoteUrl + " " + discovery.getVersion());
            }
            catch (IOException e) {
                throw new SQLException("Error connecting to Neo4j Server at " + connectionUrl, e);
            }
        }
        return this.getDriver().createExecutor(connectionUrl, this.properties);
    }

    private String getPassword() {
        return this.properties.getProperty("password");
    }

    private String getUser() {
        return this.properties.getProperty("user");
    }

    private boolean hasAuth() {
        return this.properties.contains("user") && this.properties.contains("password");
    }

    private boolean hasDebug() {
        return Connections.hasDebug(this.properties);
    }

    private void validateVersion() throws SQLException {
        if (this.version.getMajorVersion() < 1 || this.version.getMajorVersion() == 1 && this.version.getMinorVersion() < 5) {
            throw new SQLException("Unsupported Neo4j version:" + this.version);
        }
    }

    @Override
    public Statement createStatement() throws SQLException {
        return this.debug(new Neo4jStatement(this));
    }

    @Override
    public PreparedStatement prepareStatement(String statement) throws SQLException {
        return this.debug(new Neo4jPreparedStatement(this, statement));
    }

    @Override
    public void setAutoCommit(boolean autoCommit) throws SQLException {
        if (autoCommit == this.autoCommit) {
            return;
        }
        if (autoCommit) {
            this.commit();
        }
        this.autoCommit = autoCommit;
    }

    @Override
    public boolean getAutoCommit() throws SQLException {
        return this.autoCommit;
    }

    @Override
    public void commit() throws SQLException {
        this.checkClosed("commit");
        if (this.autoCommit) {
            throw new SQLException("Commit called on auto-committed connection");
        }
        try {
            this.queryExecutor.commit();
        }
        catch (SQLException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SQLException("Error during commit", e);
        }
    }

    @Override
    public void rollback() throws SQLException {
        this.checkClosed("rollback");
        if (this.getAutoCommit()) {
            throw new SQLException("Rollback called on auto-committed connection");
        }
        try {
            this.queryExecutor.rollback();
        }
        catch (SQLException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SQLException("Error during commit", e);
        }
    }

    @Override
    public void close() throws SQLException {
        this.checkClosed("close");
        try {
            this.queryExecutor.rollback();
            this.queryExecutor.stop();
        }
        catch (Exception e) {
            e.printStackTrace();
        }
        finally {
            this.closed = true;
        }
    }

    @Override
    public boolean isClosed() throws SQLException {
        return this.closed;
    }

    @Override
    public DatabaseMetaData getMetaData() throws SQLException {
        return this.debug(new Neo4jDatabaseMetaData(this));
    }

    @Override
    public void setReadOnly(boolean readOnly) throws SQLException {
        this.readonly = readOnly;
    }

    @Override
    public boolean isReadOnly() throws SQLException {
        return this.readonly;
    }

    @Override
    public SQLWarning getWarnings() throws SQLException {
        return this.sqlWarnings;
    }

    @Override
    public void clearWarnings() throws SQLException {
        this.sqlWarnings = null;
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency) throws SQLException {
        return this.debug(new Neo4jStatement(this));
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency) throws SQLException {
        return this.debug(new Neo4jPreparedStatement(this, this.nativeSQL(sql)));
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        return this.debug(new Neo4jStatement(this));
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int resultSetType, int resultSetConcurrency, int resultSetHoldability) throws SQLException {
        return this.debug(new Neo4jPreparedStatement(this, this.nativeSQL(sql)));
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys) throws SQLException {
        return this.debug(new Neo4jPreparedStatement(this, this.nativeSQL(sql)));
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int[] columnIndexes) throws SQLException {
        return this.debug(new Neo4jPreparedStatement(this, this.nativeSQL(sql)));
    }

    @Override
    public PreparedStatement prepareStatement(String sql, String[] columnNames) throws SQLException {
        return this.debug(new Neo4jPreparedStatement(this, this.nativeSQL(sql)));
    }

    @Override
    public boolean isValid(int timeout) throws SQLException {
        return true;
    }

    public ResultSet executeQuery(String query, Map<String, Object> parameters) throws SQLException {
        this.checkClosed("execute");
        this.checkReadOnly(query);
        try {
            if (log.isDebugEnabled()) {
                log.debug("Executing query: " + query + "\n with params " + parameters);
            }
            ExecutionResult result = this.queryExecutor.executeQuery(query, parameters, this.autoCommit);
            return this.debug(this.toResultSet(result));
        }
        catch (SQLException e) {
            throw e;
        }
        catch (Exception e) {
            throw new SQLException("Error executing query " + query + "\n with params " + parameters, e);
        }
    }

    private void checkClosed(String method) throws SQLException {
        if (this.isClosed()) {
            throw new SQLException(method + " called on closed connection.");
        }
    }

    private void checkReadOnly(String query) throws SQLException {
        if (this.readonly && this.isMutating(query)) {
            throw new SQLException("Mutating Query in readonly mode: " + query);
        }
    }

    private boolean isMutating(String query) {
        return query.matches("(?is).*\\b(create|relate|delete|set)\\b.*");
    }

    public String tableColumns(String tableName, String columnPrefix) throws SQLException {
        ResultSet columns = this.executeQuery(this.driver.getQueries().getColumns(tableName), Collections.singletonMap("typeName", tableName));
        StringBuilder columnsBuilder = new StringBuilder();
        while (columns.next()) {
            if (columnsBuilder.length() > 0) {
                columnsBuilder.append(',');
            }
            columnsBuilder.append(columnPrefix).append(columns.getString("property.name"));
        }
        return columnsBuilder.toString();
    }

    public Iterable<String> returnProperties(String tableName, String columnPrefix) throws SQLException {
        ResultSet columns = this.executeQuery(this.driver.getQueries().getColumns(tableName), Collections.singletonMap("typeName", tableName));
        ArrayList<String> properties = new ArrayList<String>();
        while (columns.next()) {
            properties.add(columnPrefix + columns.getString("property.name"));
        }
        return properties;
    }

    String getURL() {
        return this.url;
    }

    protected ResultSet toResultSet(ExecutionResult result) throws SQLException {
        return new IteratorResultSet(this, result.columns(), result.getResult());
    }

    public <T> T debug(T obj) {
        return Connections.debug(obj, this.debug);
    }

    public Properties getProperties() {
        return this.properties;
    }

    public Driver getDriver() {
        return this.driver;
    }

    public Version getVersion() {
        return this.version;
    }
}

