(ns splendid.jfx
  (:import (splendid SplendidJFX)
           (java.net MalformedURLException URL)
           (javafx.application Platform)
           (javafx.event EventHandler)
           javafx.fxml.FXMLLoader
           (javafx.scene Node Parent Scene)
           (javafx.scene.layout FlowPane GridPane HBox Pane VBox)
           (javafx.stage Stage)))

(defn launch-application
  "Launches a JFX app. The `init-fn` will be called with the primary Stage."
  [^clojure.lang.IFn init-fn]
  (SplendidJFX/launchApplication init-fn))

(defn launch-empty-application
  "Opens a JFX Stage and returns it. Useful during development."
  []
  (let [s (promise)]
   (SplendidJFX/launchApplication
    (fn [^Stage stage]
      (.setScene stage (Scene. (Pane.), 640.0, 480.0))
      (.show stage)
      (deliver s stage)))
   @s))

(defmacro jfx
  "Runs `body` on the JavaFX Application Thread and blocks until execution has
  finished and returns its result."
  [& body]
  `(let [f# (fn [] ~@body)
         p# (promise)]
     (if (Platform/isFxApplicationThread)
       (f#)
       (do
         (Platform/runLater #(deliver p# (f#)))
         @p#))))

(defn update-stage
  "A utility function during development to test UIs.
  Takes a layout `pane` and updates the primary Stage to display it."
  [^Parent pane]
  (when SplendidJFX/stage
    (jfx (.setScene SplendidJFX/stage (Scene. pane)))))

(defn load-fxml
  "Loads the .fxml file `f`, returns the top-level container in that file,
  with all its components."
  [f]
  (FXMLLoader/load (try (URL. (str f))
                        (catch MalformedURLException e
                          (URL. (str "file:////" f))))))

(defmacro defhandler
  "Takes an `event` which denotes the event for which you want to implement
  a handler for a given `node`."
  [event ^Node node & body]
  (let [f (symbol (str ".setOn" (subs (name event) 2)))]
    `(~f ~node
         (reify EventHandler
                (handle [this# event#] ~@body)))))

(defmacro defhandler*
  "Like `defhandler`, but takes `fn` of two arguments: this and the event."
  [event ^Node node fn]
  (let [f (symbol (str ".setOn" (subs (name event) 2)))]
    `(~f ~node
         (reify EventHandler
           (handle [this# event#] (~fn this# event#))))))
