(ns splendid.jfx
  (:import (splendid SplendidJFX)
           (java.net MalformedURLException URL)
           (javafx.application Platform)
           (javafx.beans.property BooleanProperty DoubleProperty
                                  FloatProperty IntegerProperty
                                  LongProperty ObjectProperty StringProperty)
           javafx.beans.binding.Bindings
           (javafx.event EventHandler)
           javafx.fxml.FXMLLoader
           (javafx.scene Group Node Parent Scene)
           javafx.scene.control.Accordion
           (javafx.scene.layout FlowPane GridPane HBox Pane VBox)
           (javafx.stage Stage)))

(defonce ^Stage primary-stage
  (let [s (promise)]
    (SplendidJFX/launchApplication #(deliver s %))
    @s))

(defmacro jfx
  "Runs `body` on the JavaFX Application Thread and blocks until execution has
  finished and returns its result."
  [& body]
  `(let [f# (fn [] ~@body)
         p# (promise)]
     (if (Platform/isFxApplicationThread)
       (f#)
       (do
         (Platform/runLater #(deliver p# (f#)))
         @p#))))

(defprotocol Showable
  "Objects that can be shown on a javafx.stage.Stage."
  (show* [obj stage]
    "Displays `obj` on the given Stage `stage`."))

(extend-protocol Showable
  clojure.lang.Fn
  (show* [f stage]
    (jfx (f stage)))
  Parent
  (show* [p, ^Stage stage]
    (jfx
     (.setScene stage (Scene. p))
     (.show stage)))
  java.util.Map
  (show* [m stage]
    (show* (get m :ui) stage)) ; calling show* with the Parent `(:ui m)`
  )

(defn show
  "Takes a Showable object `s` and displays it on the Stage `stage`.
  If no Stage is given, the `primary-stage` is used by default."
  ([s] (show* s primary-stage))
  ([stage, s] (show* s stage)))

(defn jfx-class? [^Class c]
  (cond
   (symbol? c) (jfx-class? (resolve c))
   (class? c)  (boolean (.startsWith (.getName c) "javafx."))
   :else false))

(defn jfx-instance? [obj]
  (jfx-class? (class obj)))

;; ## FXML

(defn scenegraph
  "Takes a Node `x`, typically a layout (something that can have children),
  and returns the scenegraph from this point on, including `x`,
  represented as a lazy seq."
  [^Node x]
  (lazy-cat
   [x]
   (when (instance? Parent x)
     (map scenegraph (.getChildrenUnmodifiable ^Parent x)))))

(defn collect-ids
  "Takes a seq of `nodes` (i.e. a flattended scenegraph, see `flatten-scenegraph`)
  and returns a map of node ID to node. For example: a Button `b` with the ID
  `ok-bt` would be returned as `{:ok-bt b}`."
  [nodes]
  (reduce #(let [id (keyword (.getId ^Node %2))]
             (cond
              (nil? id) %1 ; the control has no id, so we ignore it

              (contains? %1 id) ; ==>
              (throw
               (Exception.
                (format "ID `%s` was assigned at least twice to a control." id)))

              :else (assoc %1 id %2) ; keep this id/control pair
              ))
          (sorted-map)
          nodes))

(defn load-fxml
  "Loads the FXML file `f` and returns a hashmap `h`, containing the top-level
  node of `f` (typically a layout with its components), mapped to the key `:ui`.
  `h` also contains all components that were assigned an ID in `f`.
  You may not use the ID `\"ui\"` in your FXML files if you want to load them
  via `load-fxml`."
  [f]
  (let [fxml (FXMLLoader/load (try (URL. (str f))
                                   (catch MalformedURLException e
                                     (URL. (str "file:////" f)))))]
    (-> fxml
        scenegraph
        flatten
        collect-ids
        (assoc :ui fxml))))

;; Okay, this section will provide some macro magic + eval.
;; In general eval is perceived as evil, and this feature may
;; get removed in future versions.
;; But for now I see this experimental feature as beneficial,
;; so here it is.
(defmacro import-fxml
  "Will load the specified FXML `files` in the given order at macro expansion time.
  It will expand into code that wraps the `body` in a `let`, where all JFX
  components, which were assigend an ID in the corresponding FXML file,
  are automagically extracted/destructured.
  On top of that they are also type-hinted for your convenience.
  So the `body` may refer to the components and enjoy reflection free calls.
  This macro is public because it will appear in expansions of `with-fxml`.
  But this is private code. Don’t call!"
  [files & body]
  (let [components (apply merge (map load-fxml files))
        bindings (mapcat (fn [[k v]]
                           [(with-meta (symbol (name k))
                                       {:tag (symbol (.getName (class v)))})
                            `(~k ~'fxml)])
                         components)]
    `(let [~'fxml (apply merge (map load-fxml ~files))
           ~@bindings]
       ~@body)))

(defmacro with-fxml
  "Loads the specified FXML `files` in the given order, and makes all components
  that were assigned an ID available in the `body`, type-hinted. Make sure to
  specify your topmost container last, so that it will be available as `ui`.
  In the body the symbol `fxml` provides a map of all merged components from
  the `files`, including the last toplevel container, under the key `:ui`.

  Warning: this is an experimental feature, and it is not certain that it will
  make it into the 1.0.0 release."
  [files & body]
  `(eval '(import-fxml ~files ~@body)))


;; ## Containers

(defprotocol Container
  (add [pane elements] [pane elements obj]
   "Adds elements to the container. Optionally takes an obj which may be used
  by specific Containers (i.e. MigLayout)."))

(defn- add-template
  "Serves as a template for implementing the `add` function.
  Private code. Don’t call."
  []
  `(~'add [~'pane ~'elements]
     (if (jfx-instance? ~'elements)
       (.add (.getChildren ~'pane) ~'elements)
       (.addAll (.getChildren ~'pane) (to-array ~'elements)))))

(extend-protocol Container
  Pane #=(add-template) ; expanding `default-add` during read time
  GridPane
  (add
    ([pane elements]
       (if (jfx-instance? elements)
         (.add (.getChildren pane) elements)
         (.addAll (.getChildren pane) (to-array elements))))
    ([pane element [column row colspan rowspan]]
       (let [colspan (or colspan 0)
             rowspan (or rowspan 0)]
         (.add pane element column row colspan rowspan))))
  Group #=(add-template)
  Scene (add [scene root-pane] (.setRoot scene root-pane))
  Stage (add [stage scene] (.setScene stage scene))
  Accordion
  (add [accordion elements]
    (if (instance? Node elements)
      (.add (.getPanes accordion) elements)
      (.addAll (.getPanes accordion) (to-array elements))))
  )


;; ## Events

(defmacro defhandler
  "Takes an `event` which denotes the event for which you want to implement
  a handler for a given `node`."
  [event ^Node node & body]
  (let [f (symbol (str ".setOn" (subs (name event) 2)))]
    `(~f ~node
         (reify EventHandler
                (handle [this# event#] ~@body)))))

(defmacro defhandler*
  "Like `defhandler`, but takes `fn` of two arguments: this and the event."
  [event ^Node node fn]
  (let [f (symbol (str ".setOn" (subs (name event) 2)))]
    `(~f ~node
         (reify EventHandler
           (handle [this# event#] (~fn this# event#))))))


;; ## Bindings

(defprotocol Binding
  (bind [property properties f]
    "Creates a binding for `property` which observes changes in one or more
  `properties` and runs the zero-arity function `f` when at least one of the
  `properties` changed.
  The return value of `f` will be the new value of `property`."))

(defn- binding-template
  "Template that serves for nearly identical code, to implement bindings.
  Private code, don’t call."
  [name]
  (let [method-name (symbol (str "Bindings/create" name "Binding"))
        property-name (symbol (str name "Property"))]
    `(~'bind [~'p ~'observables ~'f]
       (.bind ~'p (~method-name ~'f (into-array ~property-name  ~'observables))))))

(extend-protocol Binding
  BooleanProperty #=(binding-template Boolean)
  FloatProperty   #=(binding-template Float)
  DoubleProperty  #=(binding-template Double)
  IntegerProperty #=(binding-template Integer)
  LongProperty    #=(binding-template Long)
  StringProperty  #=(binding-template String)
  ObjectProperty  #=(binding-template Object))

(defmacro defbinding
  "Like `bind`, only that you can provide a body `expr` instead of a function."
  [property properties & expr]
  `(bind ~property ~properties (fn [] ~@expr)))
