/*
 * Decompiled with CFR 0.152.
 */
package org.httpkit.client;

import java.io.IOException;
import java.net.InetSocketAddress;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.nio.ByteBuffer;
import java.nio.channels.SelectionKey;
import java.nio.channels.Selector;
import java.nio.channels.SocketChannel;
import java.security.NoSuchAlgorithmException;
import java.util.Iterator;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLEngine;
import javax.net.ssl.SSLException;
import org.httpkit.DynamicBytes;
import org.httpkit.HTTPException;
import org.httpkit.HeaderMap;
import org.httpkit.HttpMethod;
import org.httpkit.HttpUtils;
import org.httpkit.PriorityQueue;
import org.httpkit.ProtocolException;
import org.httpkit.client.HttpsRequest;
import org.httpkit.client.IRespListener;
import org.httpkit.client.PersistentConn;
import org.httpkit.client.Request;
import org.httpkit.client.RequestConfig;
import org.httpkit.client.State;
import org.httpkit.client.TimeoutException;

public final class HttpClient
implements Runnable {
    private static final AtomicInteger ID = new AtomicInteger(0);
    public static final SSLContext DEFAULT_CONTEXT;
    private final Queue<Request> pending = new ConcurrentLinkedQueue<Request>();
    private final PriorityQueue<Request> requests = new PriorityQueue();
    private final PriorityQueue<PersistentConn> keepalives = new PriorityQueue();
    private volatile boolean running = true;
    private final ByteBuffer buffer = ByteBuffer.allocateDirect(65536);
    private final Selector selector;

    public HttpClient() throws IOException {
        int id = ID.incrementAndGet();
        String name = "client-loop";
        if (id > 1) {
            name = name + "#" + id;
        }
        this.selector = Selector.open();
        Thread t = new Thread((Runnable)this, name);
        t.setDaemon(true);
        t.start();
    }

    private void clearTimeout(long now) {
        PersistentConn pc;
        Request r;
        while ((r = this.requests.peek()) != null && r.isTimeout(now)) {
            String msg = "connect timeout: ";
            if (r.isConnected) {
                msg = "read timeout: ";
            }
            r.finish(new TimeoutException(msg + r.cfg.timeout + "ms"));
            if (r.key == null) continue;
            this.closeQuietly(r.key);
        }
        while ((pc = this.keepalives.peek()) != null && pc.isTimeout(now)) {
            this.closeQuietly(pc.key);
            this.keepalives.poll();
        }
    }

    private boolean cleanAndRetryIfBroken(SelectionKey key, Request req) {
        this.closeQuietly(key);
        this.keepalives.remove(key);
        if (req.isReuseConn && req.decoder.state == State.READ_INITIAL) {
            for (ByteBuffer b : req.request) {
                b.position(0);
            }
            req.isReuseConn = false;
            this.requests.remove(req);
            this.pending.offer(req);
            this.selector.wakeup();
            return true;
        }
        return false;
    }

    private void doRead(SelectionKey key, long now) {
        Request req = (Request)key.attachment();
        SocketChannel ch = (SocketChannel)key.channel();
        int read = 0;
        try {
            this.buffer.clear();
            if (req instanceof HttpsRequest) {
                HttpsRequest httpsReq = (HttpsRequest)req;
                read = httpsReq.handshaken ? httpsReq.unwrapRead(this.buffer) : httpsReq.doHandshake(this.buffer);
            } else {
                read = ch.read(this.buffer);
            }
        }
        catch (IOException e) {
            if (!this.cleanAndRetryIfBroken(key, req)) {
                req.finish(e);
            }
        }
        catch (Exception e) {
            req.finish(e);
        }
        if (read == -1) {
            if (!this.cleanAndRetryIfBroken(key, req)) {
                req.finish();
            }
        } else if (read > 0) {
            req.onProgress(now);
            this.buffer.flip();
            try {
                if (req.decoder.decode(this.buffer) == State.ALL_READ) {
                    req.finish();
                    if (req.cfg.keepAlive > 0) {
                        this.keepalives.offer(new PersistentConn(now + (long)req.cfg.keepAlive, req.addr, key));
                    } else {
                        this.closeQuietly(key);
                    }
                }
            }
            catch (HTTPException e) {
                this.closeQuietly(key);
                req.finish(e);
            }
            catch (Exception e) {
                this.closeQuietly(key);
                req.finish(e);
                HttpUtils.printError("should not happen", e);
            }
        }
    }

    private void closeQuietly(SelectionKey key) {
        try {
            key.channel().close();
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    private void doWrite(SelectionKey key) {
        Request req = (Request)key.attachment();
        SocketChannel ch = (SocketChannel)key.channel();
        try {
            if (req instanceof HttpsRequest) {
                HttpsRequest httpsReq = (HttpsRequest)req;
                if (httpsReq.handshaken) {
                    httpsReq.writeWrappedRequest();
                } else {
                    this.buffer.clear();
                    if (httpsReq.doHandshake(this.buffer) < 0) {
                        req.finish();
                    }
                }
            } else {
                ByteBuffer[] buffers = req.request;
                ch.write(buffers);
                if (!buffers[buffers.length - 1].hasRemaining()) {
                    key.interestOps(1);
                }
            }
        }
        catch (IOException e) {
            if (!this.cleanAndRetryIfBroken(key, req)) {
                req.finish(e);
            }
        }
        catch (Exception e) {
            req.finish(e);
        }
    }

    public void exec(String url, RequestConfig cfg, SSLEngine engine, IRespListener cb) {
        ByteBuffer[] request;
        InetSocketAddress addr;
        URI uri;
        try {
            uri = new URI(url);
        }
        catch (URISyntaxException e) {
            cb.onThrowable(e);
            return;
        }
        if (uri.getHost() == null) {
            cb.onThrowable(new IllegalArgumentException("host is null: " + url));
            return;
        }
        String scheme = uri.getScheme();
        if (!"http".equals(scheme) && !"https".equals(scheme)) {
            String message = scheme == null ? "No protocol specified" : scheme + " is not supported";
            cb.onThrowable(new ProtocolException(message));
            return;
        }
        try {
            addr = HttpUtils.getServerAddr(uri);
        }
        catch (UnknownHostException e) {
            cb.onThrowable(e);
            return;
        }
        HeaderMap headers = HeaderMap.camelCase(cfg.headers);
        headers.put("Host", HttpUtils.getHost(uri));
        if (!headers.containsKey("Accept")) {
            headers.put("Accept", "*/*");
        }
        if (!headers.containsKey("User-Agent")) {
            headers.put("User-Agent", RequestConfig.DEFAULT_USER_AGENT);
        }
        if (!headers.containsKey("Accept-Encoding")) {
            headers.put("Accept-Encoding", "gzip, deflate");
        }
        try {
            request = this.encode(cfg.method, headers, cfg.body, uri);
        }
        catch (IOException e) {
            cb.onThrowable(e);
            return;
        }
        if ("https".equals(scheme)) {
            if (engine == null) {
                engine = DEFAULT_CONTEXT.createSSLEngine();
            }
            engine.setUseClientMode(true);
            this.pending.offer(new HttpsRequest(addr, request, cb, this.requests, cfg, engine));
        } else {
            this.pending.offer(new Request(addr, request, cb, this.requests, cfg));
        }
        this.selector.wakeup();
    }

    private ByteBuffer[] encode(HttpMethod method, HeaderMap headers, Object body, URI uri) throws IOException {
        ByteBuffer bodyBuffer = HttpUtils.bodyBuffer(body);
        if (body != null) {
            headers.put("Content-Length", Integer.toString(bodyBuffer.remaining()));
        } else {
            headers.put("Content-Length", "0");
        }
        DynamicBytes bytes = new DynamicBytes(196);
        bytes.append(method.toString()).append((byte)32).append(HttpUtils.getPath(uri));
        bytes.append(" HTTP/1.1\r\n");
        headers.encodeHeaders(bytes);
        ByteBuffer headBuffer = ByteBuffer.wrap(bytes.get(), 0, bytes.length());
        if (bodyBuffer == null) {
            return new ByteBuffer[]{headBuffer};
        }
        return new ByteBuffer[]{headBuffer, bodyBuffer};
    }

    private void finishConnect(SelectionKey key, long now) {
        SocketChannel ch = (SocketChannel)key.channel();
        Request req = (Request)key.attachment();
        try {
            if (ch.finishConnect()) {
                req.isConnected = true;
                req.onProgress(now);
                key.interestOps(4);
                if (req instanceof HttpsRequest) {
                    ((HttpsRequest)req).engine.beginHandshake();
                }
            }
        }
        catch (IOException e) {
            this.closeQuietly(key);
            req.finish(e);
        }
    }

    private void processPending() {
        Request job = null;
        while ((job = this.pending.poll()) != null) {
            PersistentConn con;
            if (job.cfg.keepAlive > 0 && (con = this.keepalives.remove(job.addr)) != null) {
                SelectionKey key = con.key;
                if (key.isValid()) {
                    job.isReuseConn = true;
                    try {
                        job.recycle((Request)key.attachment());
                        key.attach(job);
                        key.interestOps(4);
                        this.requests.offer(job);
                        continue;
                    }
                    catch (SSLException e) {
                        this.closeQuietly(key);
                    }
                } else {
                    this.closeQuietly(key);
                }
            }
            try {
                SocketChannel ch = SocketChannel.open();
                ch.configureBlocking(false);
                job.key = ch.register(this.selector, 8, job);
                ch.connect(job.addr);
                this.requests.offer(job);
            }
            catch (IOException e) {
                job.finish(e);
                HttpUtils.printError("Try to connect " + job.addr, e);
            }
        }
    }

    @Override
    public void run() {
        while (this.running) {
            try {
                long now = System.currentTimeMillis();
                int select = this.selector.select(2000L);
                if (select > 0) {
                    Set<SelectionKey> selectedKeys = this.selector.selectedKeys();
                    Iterator<SelectionKey> ite = selectedKeys.iterator();
                    while (ite.hasNext()) {
                        SelectionKey key = ite.next();
                        if (!key.isValid()) continue;
                        if (key.isConnectable()) {
                            this.finishConnect(key, now);
                        } else if (key.isReadable()) {
                            this.doRead(key, now);
                        } else if (key.isWritable()) {
                            this.doWrite(key);
                        }
                        ite.remove();
                    }
                }
                this.clearTimeout(now);
                this.processPending();
            }
            catch (IOException e) {
                HttpUtils.printError("select exception", e);
            }
        }
    }

    public void stop() throws IOException {
        this.running = false;
        if (this.selector != null) {
            this.selector.close();
        }
    }

    public String toString() {
        return this.getClass().getCanonicalName();
    }

    static {
        try {
            DEFAULT_CONTEXT = SSLContext.getDefault();
        }
        catch (NoSuchAlgorithmException e) {
            throw new Error("Failed to initialize SSLContext", e);
        }
    }
}

