(ns stigmergy.ikota
  (:refer-clojure :exclude [format])
  (:require [clojure.string]
            [clojure.walk :as w]
            #?(:cljs [goog.string :as gstring])))

(defn primitive? [hiccup]
  (or (string? hiccup)
      (number? hiccup)
      (keyword? hiccup)
      (boolean? hiccup)
      (inst? hiccup)
      (nil? hiccup)
      (char? hiccup)))

(defn component? [c]
  (and (sequential? c)
       (let [first-element (first c)]
         (or (fn? first-element)
             (map? first-element)))))

(defn normalize-hiccup
  "hiccup should be in form of [:tag {} body]"
  [hiccup]
  (let [hiccup (w/postwalk #(if (nil? %)
                              ""
                              %)
                           hiccup)
        [tag maybe-attr-map & body] hiccup
        nh (cond
             (= 1 (count hiccup))              (conj hiccup {})
             (and (not (map? maybe-attr-map))
                  (nil? body))                 (let [element maybe-attr-map]
                                                 [tag {} element])
             
             (and (not (map? maybe-attr-map))) (let [element maybe-attr-map]
                                                 (into [tag {} element] body)) 
             :else hiccup)]
    nh))

(defn extract-tag-id-css-classes [tag-maybe-id-css-classes]
  (let [as-str (name tag-maybe-id-css-classes)
        chunks (clojure.string/split as-str #"\.")
        tag-with-maybe-id (first chunks)
        [tag id] (clojure.string/split tag-with-maybe-id #"#")
        css-classes (rest chunks)]
    [tag id css-classes]))

(defn format [fmt & args]
  #?(:cljs
     (apply gstring/format fmt args))
  #?(:clj
     (apply clojure.core/format fmt args)))



(defn style-map->css-str [m]
  (->> m
       (map (fn [[k v]]
              (let [css-key (name k)
                    css-val (cond
                              (keyword? v) (name v)
                              (number? v) (str v "px")
                              :else v)]
                (format "%s: %s" css-key css-val))))
       (clojure.string/join "; ")))

(defn- attribute-map->str [attribute-map]
  (clojure.string/join " " (for [[k v] attribute-map]
                             (if (= k :style)
                               (format "style=\"%s\"" (style-map->css-str v))
                               (format "%s=\"%s\"" (name k) v)))))


(declare hiccup->html-str)

(defn- create-html-str [tag-maybe-id-css-classes attr-map hiccup-content]
  (let [[tag-name id css-classes] (extract-tag-id-css-classes tag-maybe-id-css-classes)
        attribute-map (if (empty? css-classes)
                        attr-map
                        (assoc attr-map :class (clojure.string/join " " css-classes)))
        attribute-map (if (nil? id)
                        attribute-map
                        (assoc attribute-map :id id))
        attributes (attribute-map->str attribute-map)
        nested? (and (sequential? hiccup-content)
                     (= 1 (count hiccup-content)))
        hiccup-content (if nested?
                         (first hiccup-content)
                         hiccup-content)]
    (cond
      (or (= tag-name "link")
          (= tag-name "input")) (format "<%s %s>" tag-name attributes)
      :else (let [begin-tag (format "<%s %s>" tag-name attributes)
                  body (->> hiccup-content hiccup->html-str)
                  end-tag (format "</%s>" tag-name) 
                  well-formed [begin-tag body end-tag]
                  html-str (clojure.string/join " " well-formed)]
              html-str))))

(declare  component-node->hiccup)

(defn fn->hiccup [render-fn params]
  (let [component-or-hiccup (apply render-fn params)
        hiccup (if (component? component-or-hiccup)
                 (component-node->hiccup component-or-hiccup)
                 (normalize-hiccup component-or-hiccup))]
    (w/prewalk component-node->hiccup
               hiccup)))

(defn hiccup->html-str [hiccup]
  (if (primitive? hiccup)
    (str hiccup)
    (let [first-element (first hiccup)
          second-element (second hiccup)]
      (cond
        (and (keyword? first-element)
             (map? second-element)) (let [tag first-element
                                          attr-map second-element
                                          content (drop 2 hiccup)]
                                      (create-html-str tag attr-map content))
        (keyword? first-element) (let [tag first-element
                                       content (rest hiccup)]
                                   (create-html-str tag {} content))
        (fn? first-element) (let [func first-element
                                  params (rest hiccup)
                                  h (fn->hiccup func params)]
                              (hiccup->html-str h))
        (map? first-element) (let [func (:reagent-render first-element)
                                   params (rest hiccup)
                                   h (fn->hiccup func params)]
                               (hiccup->html-str h))
        (every? empty? hiccup) ""
        :else (clojure.string/join " " (map hiccup->html-str hiccup))))))

(defn component-node->hiccup [node]
  (if (component? node)
    (let [render-fn (let [first-element (first node)]
                      (cond
                        (fn? first-element) first-element
                        (map? first-element) (:reagent-render first-element)))
          params (rest node)
          ;;TODO: should check mounted-components first
          hiccup (fn->hiccup render-fn params)
          ]
      hiccup)
    node))

(defn component->hiccup [[{:keys [reagent-render]} & params :as normalized-component]]
  (let [hiccup (fn->hiccup reagent-render params)
        hiccup (w/prewalk component-node->hiccup
                          hiccup)]
    hiccup))

#?(:cljs
   (declare hiccup->dom))

#?(:cljs
   (defn fn->dom [render-fn params]
     (let [hiccup (fn->hiccup render-fn params)]
       (hiccup->dom hiccup))))

(defn keyword->str [kw]
  (if (keyword? kw)
    (name kw)
    kw))

#?(:cljs
   (defn create-element [tag-maybe-id-css-classes attr-map hiccup-content]
     (let [[tag-name id css-classes] (extract-tag-id-css-classes tag-maybe-id-css-classes)
           element (js/document.createElement tag-name)
           element-children (hiccup->dom hiccup-content)
           nested? (and (seq? element-children)
                        (every? seq? element-children))
           element-children (if nested?
                              (mapcat identity element-children)
                              element-children)]
       (when id
         (set! (.-id element) id))
       (when-not (empty? css-classes)
         (doseq [c css-classes]
           (.. element -classList (add c))))
       (when-not (empty? attr-map)
         (doseq [[k v] attr-map]
           (cond
             (= k :style) (.. element (setAttribute "style" (style-map->css-str v)))
             (re-find #"on-\w+-*\w+" (name k)) (let [event (clojure.string/replace (name k) "on-" "")
                                                     event (clojure.string/replace event "-" "")]
                                                 (.. element (addEventListener event v)))
             :else (let [k (keyword->str k)
                         v (keyword->str v)]
                     (.. element (setAttribute k v))))))

       (if (seq? element-children)
         (doseq [child element-children]
           (.. element (appendChild child)))
         (.. element (appendChild element-children)))
       element)))

#?(:cljs
   (defn hiccup->dom [hiccup]
     (if (primitive? hiccup)
       (js/document.createTextNode hiccup)
       (let [first-element (first hiccup)
             second-element (second hiccup)]
         (cond
           (and (keyword? first-element)
                (map? second-element)) (let [tag first-element
                                             attr-map second-element
                                             content (rest (rest hiccup))]
                                         (create-element tag attr-map content))
           (keyword? first-element) (let [tag first-element
                                          content (rest hiccup)]
                                      (create-element tag {} content))
           (fn? first-element) (let [func first-element
                                     params (rest hiccup)]
                                 (fn->dom func params))
           (map? first-element) (let [func (:reagent-render first-element)
                                      params (rest hiccup)]
                                  (fn->dom func params))
           :else (map hiccup->dom hiccup))))))

(comment

  (hiccup->html-str [:h1 "hello world"])
  (hiccup->dom [:h1 "wassup world!"]) ;; clojurescript only
  
  )
