(ns stock-grok.util.messaging.kafka
  (:require
   [stock-grok.util.messaging.contract :as contract]
   [jackdaw.client :as jc]
   [jackdaw.serdes :as js]
   [jackdaw.serdes.fn :as jsfn]
   [jackdaw.client.log :as jl]
   [taoensso.nippy :as nippy]
   [clojure.core.async :refer [put! thread close! chan] :as a]
   [clojure.spec.alpha :as s])
  (:import org.apache.kafka.common.serialization.Serdes))


(defn- edn-serializer []
  (jsfn/new-serializer {:serialize (fn [_ _ data]
                                     (when data
                                       (nippy/freeze data)))}))

(defn- edn-deserializer []
  (jsfn/new-deserializer {:deserialize (fn [_ _ data]
                                         (nippy/thaw data))}))

(defn- edn-serde []
  (Serdes/serdeFrom (edn-serializer) (edn-deserializer)))


(defn- create-producer [producer-config]
  (jc/producer producer-config
               {:key-serde   (js/string-serde)
                :value-serde (edn-serde)}))


(defn- create-consumer [consumer-config]
  (jc/consumer consumer-config
               {:key-serde   (js/string-serde)
                :value-serde (edn-serde)}))

(defn log
  "Given a consumer, returns a lazy sequence of datafied consumer records.

  If fuse-fn was provided, stops after fuse-fn returns false."
  ([^Consumer consumer polling-interval-ms]
   (log consumer polling-interval-ms (constantly true)))
  ([^Consumer consumer polling-interval-ms fuse-fn]
   (let [r (jc/poll consumer polling-interval-ms)]
     (if (fuse-fn r)
       (do (log/debugf "Got %d records" (count r))
           (lazy-cat r (log consumer polling-interval-ms fuse-fn)))
       r))))

(defn- start-consumer! [check-closed consumer timeout topics chan]
  (thread
    (jc/subscribe consumer topics)
    (doseq [item (jl/log consumer timeout check-closed)]
      (put! chan item))
    (close! chan)
    (.close consumer)))


(defn- fuse-fn [closed-atom]
  (fn [_] (not @closed-atom)))


(s/def ::bootstrap.servers string?)
(s/def ::client.id string?)
(s/def ::group.id string?)
(s/def ::enable.auto.commit boolean?)
(s/def ::kconfig
  (s/keys
   :req-un [::bootstrap.servers ::client.id]
   :opt-un [::group.id ::enable.auto.commit]))

(s/def ::topic-name string?)
(s/def ::topic
  (s/keys :req-un [::topic-name]))

(s/def ::timeout int?)
(s/def ::topics (s/coll-of ::topic))
(s/def ::subscribe-opts
  (s/keys :req-un [::topics]))

(defn- validate-config [kconfig]
  (if-not (s/valid? ::kconfig kconfig)
    (throw (ex-info (s/explain-str ::kconfig kconfig)
                    (s/explain-data ::kconfig kconfig)))))


(defn- validate-topic [topic]
  (if-not (s/valid? ::topic topic)
    (throw (ex-info (s/explain-str ::topic topic)
                    (s/explain-data ::topic topic)))))


(defn- validate-subscriber-opts [opts]
  (if-not (s/valid? ::subscribe-opts opts)
    (throw (ex-info (s/explain-str ::subscribe-opts opts)
                    (s/explain-data ::subscribe-opts opts)))))


(defn kafka-consumer [kconfig]
  (validate-config kconfig)
  (let [started (atom false)
        closed  (atom false)]
    (reify contract/subscriber
      (subscribe! [_ opts chan]
        (validate-subscriber-opts opts)
        (if (not @started)
          (let [{:keys [timeout topics]} opts
                consumer                 (create-consumer kconfig)
                fuse-fn                  (fuse-fn closed)]
            (reset! started true)
            (reset! closed false)
            (start-consumer! fuse-fn consumer timeout topics chan)
            chan)
          (throw (Exception. "A consumer already exists"))))
      (stop-subscriber! [_]
        (reset! started false)
        (reset! closed true)))))


(defn kafka-producer [kconfig]
  (validate-config kconfig)
  (let [p (create-producer kconfig)]
    (reify contract/publisher
      (publish! [_ {:keys [topic]} value]
        (validate-topic topic)
        (jc/produce! p topic value))
      (stop-publisher! [_]
        (.close p)))))
