(ns clj-kafka.serialization.serializers
  (:require [clojure.string :as str]
            [cognitect.transit :as transit]
            [jsonista.core :as jsonista])
  (:import [java.io ByteArrayOutputStream]
           [org.apache.kafka.common.serialization Serializer]))


(deftype StringSerializer []
  Serializer
  (configure [this configs is-key?])
  (serialize [this topic data]
    (when (string? data)
      (.getBytes data)))
  (close [this]))


(deftype KeywordSerializer []
  Serializer
  (configure [this configs is-key?])
  (serialize [this topic data]
    (when (keyword? data)
      (-> (str data)
         (str/replace-first #":" "")
         (.getBytes))))
  (close [this]))


(deftype EdnSerializer []
  Serializer
  (configure [this configs is-key?])
  (serialize [this topic data]
    (when data
      (.getBytes (pr-str data))))
  (close [this]))


(deftype JsonSerializer [mapper-options]
  Serializer
  (configure [this configs is-key?])
  (serialize [this topic data]
    (when data
      (jsonista/write-value-as-bytes data (jsonista/object-mapper mapper-options))))
  (close [this]))


(deftype TransitSerializer [format writer-options]
  Serializer
  (configure [this configs is-key?])
  (serialize [this topic data]
    (when data
      (let [output-stream (ByteArrayOutputStream.)]
        (-> output-stream
           (transit/writer format writer-options)
           (transit/write data))
        (.toByteArray output-stream))))
  (close [this]))
