(ns everest.event
  (:require [everest.event.proto :as proto]))

(defn stream-slice [{:keys [direction from next stream events]}]
  {:everest.stream-slice/stream stream
   :everest.stream-slice/from from
   :everest.stream-slice/next next
   :everest.stream-slice/direction direction
   :everest.stream-slice/events events})

(defn events->stream-slice [read-direction stream-id ordered-events]
  (let [from (:everest.event/position (first ordered-events))
        to (:everest.event/position (last ordered-events))]
    (stream-slice
      {:stream stream-id
       :direction read-direction
       :events ordered-events
       :from from
       :next (when to
               (if (= read-direction :everest.read-direction/forward)
                 (proto/-next to)
                 (proto/-prev to)))})))

(defn transient-event
  "Creates a transient event.

  Usually this type of even is `put!` into the `:everest.event/this-stream` of
  the subscription stream by the application."
  [{:everest.event/keys [type data metadata timestamp]
    :or {data {}, metadata {}, timestamp (java.util.Date.)}}]
  {:everest.event/type type
   :everest.event/stream :everest.stream/transient
   :everest.event/timestamp timestamp
   :everest.event/data data
   :everest.event/metadata metadata})

(defn transient?
  "True if the given `event` is transient.

  A transient event is generated by the application. It's not stored in the
  underlying event store."
  [{:everest.event/keys [stream] :as event}]
  (identical? :everest.stream/transient stream))

(defn subscription-started []
  (transient-event {:everest.event/type :everest.subscription/started}))

(defn subscription-started?
  "True if the given `event` signifies the start of subscription."
  [{:everest.event/keys [type] :as event}]
  (and (transient? event)
       (identical? :everest.subscription/started type)))

(defn subscription-stopped []
  (transient-event {:everest.event/type :everest.subscription/stopped}))

(defn subscription-stopped?
  "True if the given `event` signifies the end of subscription."
  [{:everest.event/keys [type] :as event}]
  (and (transient? event)
       (identical? :everest.subscription/stopped type)))

(defn live-processing-started []
  (transient-event
    {:everest.event/type :everest.subscription/live-processing-started}))

(defn live-processing-started?
  "True if the given `event` signifies the start of live processing."
  [{:everest.event/keys [type] :as event}]
  (and (transient? event)
       (identical? :everest.subscription/live-processing-started type)))
