(ns sv.gloud.metadata.atom
  (:require [clj-http.client :as c]
            [again.core :as a]
            [clojure.edn :as edn]
            [clojure.tools.logging :as log]))

(def default-config
  {:base-url "http://metadata.google.internal/computeMetadata/v1"
   :parse edn/read-string
   :retry-strategy #(a/max-retries 10 (a/constant-strategy 1000))
   :sleep 10000})

(defn build-request [params]
  {:method :get
   :url (str (:base-url params)
             (:path params))
   :headers {"Metadata-Flavor" "Google"}
   :query-params {:wait_for_change (:wait-for-change params false)
                  :last_etag (:last-etag params)}})

(defn extract-data [params response]
  (try
    {:data ((:parse params) (:body response))
     :etag (get-in response [:headers "ETag"])}
    (catch Exception e
      (log/error e "can not extract metadata from: " (:path params))
      nil)))

(defn get-data [params]
  (let [params (merge default-config params)]
    (extract-data params (c/request (build-request params)))))

(defn update-data [params]
  (try
    (let [result (get-data params)]
      (when-let [data (:data result)]
        (log/info "updating metadata from:" (:path params) "to:" data)
        (reset! (:atom params) data))
      result)
    (catch Exception e
      (log/error e "failed to get metadata")
      nil)))

(defn watch-loop [params]
  (let [last-etag (atom "0")]
    (while true
      (let [result (update-data (assoc params
                                       :last-etag @last-etag
                                       :wait-for-change true))]
        (when-let [etag (:etag result)]
          (reset! last-etag etag)))
      (Thread/sleep (:sleep default-config)))))

(defn start [config]
  (future
    (watch-loop config)))

(defn stop [process]
  (future-cancel process))

(comment
  ;; Example
  
  (def example-config
    {:path "/project/attributes/config"
     :atom (atom {})})

  (def process (start example-config))

  (stop process)

  ;; extract-data example
  (let [params (merge default-config example-config)]
    (extract-data
     params
     (c/request
      (build-request params)))))
