(ns sweet-tooth.workflow.tasks
  {:boot/export-tasks true}
  (:require [adzerk
             [boot-cljs :refer [cljs]]
             [boot-reload :refer [reload]]]
            [boot
             [core :as c :refer [deftask with-pre-wrap]]
             [pod :as pod]]
            [boot.task.built-in :refer [aot jar pom repl target uber watch]]
            [environ.boot :refer [environ]]
            [clojure.java.io :as io]
            [clojure.string :as str]
            [deraen.boot-sass :as bs]
            [system.boot :as sb]))

(defn sym->var
  [sym]
  (if (symbol? sym)
    (-> sym namespace symbol find-ns (ns-resolve sym))
    sym))

;; run dev system
(deftask run
  "Start up everything needed for local dev"
  [s sys     SYS     edn "The system var"
   j js-main JS-MAIN sym "JS function to run on reload"
   f files   FILES   [str] "Files to watch for system reset"
   r regexes bool "Treat --files as regexes, not file names. Only one of regexes|paths is allowed."
   p paths   bool "Treat --files as classpath paths, not file names. Only one of regexes|paths is allowed."]
  (let [sys (sym->var sys)]
    (comp (watch)
          (repl :server true)
          (bs/sass)
          (sb/system :sys sys :auto true :files files :regexes regexes :paths paths)
          (reload :on-jsload js-main)
          (cljs :compiler-options {:asset-path "/main.out"}
                :source-map true))))

;; local dev
(deftask dev
  "Simple alias to run application in development mode"
  [p port    PORT   str "HTTP port number"
   u uri     URI    str "datomic URI"
   s sys     SYS    edn "The system var"
   j js-main JSMAIN sym "JS function to run on reload"]
  (comp (environ :env {:http-server-port port :db-uri uri})
        (run :sys sys)
        (target :dir #{"target/dev"})))

;; simple expire
(defn- expire-rewrite
  [index]
  (let [timestamp (quot (System/currentTimeMillis) 1000)]
    (str/replace index #"main.(css|js)" (str "main.$1?at=" timestamp))))

(deftask expire-assets
  "modify index.html to change URLs to expire assets"
  []
  (let [dir (c/tmp-dir!)]
    (with-pre-wrap fileset
      (let [index (first (c/by-path #{"index.html"} (c/input-files fileset)))
            file  (io/file dir "index.html")]
        (if (.exists file)
          (spit file (expire-rewrite (slurp (c/tmp-file index)))))
        (c/commit! (c/add-resource fileset dir))))))

(deftask build
  "Builds an uberjar"
  [v version VERSION str "version number"
   p project PROJECT sym "project name"
   m main    MAIN    sym "server ns with main fn"
   f file    FILE    str "name of jar file"]
  [version project main file]
  (c/merge-env! :resource-paths (c/get-env :source-paths))
  (comp (bs/sass)
        (cljs :optimizations :advanced
              :compiler-options {:parallel-build true})
        (expire-assets)
        (pom :project project :version version)
        (uber :exclude (conj pod/standard-jar-exclusions #".*\.html" #"clout/.*" #"license" #"LICENSE")) ; needed for arcane magic reasons
        (aot :namespace #{main})
        (jar :main main :file file :project project)
        (target :dir #{"target/build"})))
