;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.amalgam
  (:require
    [com.stuartsierra.component :as component]))

(defn overwrite
  "Resolve current and configuration values by always taking the configuration value."
  [_v c]
  c)

(defn configure
  "Recursively merge config map into component using merge-fns as necessary.  Maps are recursively
  merged.  Sets, vectors and lists are non-recursively concatenated.

  This behavior can be overridden with merge-fns, which should match the structure of config and
  value.  A merge function takes two arguments: the current value, and the configured value
  (respectively) and should return the final, resolved value.

  Example:

  ```clojure
  user> (amalgam/configure {:a [1] :b {:c [2]}}
          {:a [10] :b {:c [20] :d 30}}
          {:b {:c amalgam/overwrite}})
  {:a [1 10], :b {:c [20], :d 30}}
  ```"
  ([component config] (configure component config {}))
  ([component config merge-fn]
   (if (fn? merge-fn)
     (merge-fn component config)
     (cond
       (and (map? config) (map? component))
       (reduce-kv
         (fn [m k c]
           (assoc m k
             (if (contains? m k)
               (configure (get m k) c (when (and (map? merge-fn) (contains? merge-fn k))
                                        (get merge-fn k)))
               c)))
         component
         config)
       (or (and (set? config) (set? component))
         (and (vector? config) (vector? component)))
       (into component config)
       (and (list? config) (list? component))
       (apply list (concat component config))
       :else
       config))))

(defn start-system
  "Construct, configure, then start a system.

  - **`make-system`** — no argument function returning a new system
  - **`read-config`** — no argument function returning the configuration map.  The configuration map
    should mirror the structure of the system.
  - **`merge-fns`** (optional) — map mirroring the structure of both the system and configuration
    maps that contains merge functions to resolve a final value from the current value and
    configured value.

  Example:

  ```clojure
  user> (amalgam/start-system #(component/system-map :component (map->Component {:foo :bar}))
          (constantly {:component {:foo :baz}})
          {:component {:foo amalgam/overwrite}})
  {:component {:foo :baz}}
  ```

  See [[configure]]"
  ([make-system read-config]
   (start-system make-system read-config {}))
  ([make-system read-config merge-fns]
   (-> (make-system)
     (configure (read-config) merge-fns)
     component/start)))

(defn run-system
  "Start a configured system and run it until the JVM exits.

  - **`make-system`** — no argument function returning a new system
  - **`read-config`** — no argument function returning the configuration map.  The configuration map
    should mirror the structure of the system.
  - **`merge-fns`** (optional) — map mirroring the structure of both the system and configuration
    maps that contains merge functions to resolve a final value from the current value and
    configured value.

  See [[configure]] [[start-system]]"
  ([make-system read-config]
   (run-system make-system read-config {}))
  ([make-system read-config merge-fns]
   (let [running? (promise)
         system (start-system make-system read-config merge-fns)]
     (.addShutdownHook (Runtime/getRuntime) (Thread. #(deliver running? false)))
     (while (try @running? (catch InterruptedException _ false)))
     (component/stop system))))
