;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.amalgam.executors
  (:require
    [com.stuartsierra.component :as component]
    [systems.thoughtfull.desiderata :as desiderata])
  (:import
    (java.time Duration)
    (java.util.concurrent ExecutorService Executors Future RejectedExecutionHandler
      ScheduledExecutorService ScheduledFuture ScheduledThreadPoolExecutor ThreadPoolExecutor
      ThreadPoolExecutor$AbortPolicy ThreadPoolExecutor$CallerRunsPolicy
      ThreadPoolExecutor$DiscardOldestPolicy ThreadPoolExecutor$DiscardPolicy TimeUnit)))

(set! *warn-on-reflection* true)

(defn- convey-bindings?
  [{:as this :keys [convey-bindings? thread-factory]}]
  (if (contains? this :convey-bindings?)
    (boolean convey-bindings?)
    (not (desiderata/binding-conveying-thread-factory? thread-factory))))

(defn- bind-runnable
  ^Runnable [this ^Runnable task]
  (if (and (convey-bindings? this) (seq (get-thread-bindings)))
    (bound-fn []
      (.run task))
    task))

(defn- bind-callable
  ([this]
   (fn [task]
     (bind-callable this task)))
  (^Callable [this ^Callable task]
   (if (and (convey-bindings? this) (seq (get-thread-bindings)))
     (bound-fn []
       (.call task))
     task)))

(defn- bind-all
  [this tasks]
  (cond->> tasks
    (convey-bindings? this) (mapv (bind-callable this))))

(defn- executor-service!
  ^ExecutorService [{:keys [^ExecutorService executor-service]}]
  (when-not executor-service
    (throw (IllegalStateException. "ExecutorServiceComponent is not started")))
  executor-service)

(desiderata/defrecord ExecutorServiceComponent
  "An executor service that is also a component.  It can be injected as a dependency of other
  components, but also implements the ExecutorService interface.

  - **make-executor-service-fn** — a function that takes the ExecutorServiceComponent as an argument
    and creates an ExecutorService for wrapping.  Any options necessary for constructing an
    ExecutorService should be taken from the ExecutorServiceComponent.
  - **convey-bindings?** — if true then convey thread local var bindings with tasks as they are
    submitted to the wrapped ExecutorService, defaults to true.
  - **termination-wait-duration** — a *java.time.Duration* to wait for the ExecutorService to
    terminate when this component is stopped.  A duration of zero means don't wait at all.  If not
    specified, then wait indefinitely."
  [make-executor-service-fn termination-wait-duration]
  ExecutorService
  (awaitTermination
    [this timeout unit]
    (.awaitTermination (executor-service! this) timeout unit))
  (invokeAll
    [this tasks]
    (.invokeAll (executor-service! this) (bind-all this tasks)))
  (invokeAll
    [this tasks timeout unit]
    (.invokeAll (executor-service! this) (bind-all this tasks) timeout unit))
  (invokeAny
    [this tasks]
    (.invokeAny (executor-service! this) (bind-all this tasks)))
  (invokeAny
    [this tasks timeout unit]
    (.invokeAny (executor-service! this) (bind-all this tasks) timeout unit))
  (isShutdown
    [this]
    (.isShutdown (executor-service! this)))
  (isTerminated
    [this]
    (.isTerminated (executor-service! this)))
  (shutdown
    [this]
    (.shutdown (executor-service! this)))
  (shutdownNow
    [this]
    (.shutdownNow (executor-service! this)))
  (^Future submit
   [this ^Runnable task]
   (.submit (executor-service! this) (bind-runnable this task)))
  (^Future submit
   [this ^Runnable task result]
   (.submit (executor-service! this) (bind-runnable this task) result))
  (^Future submit
   [this ^Callable task]
   (.submit (executor-service! this) (bind-callable this task)))
  (execute
    [this command]
    (.execute (executor-service! this) (bind-runnable this command)))
  component/Lifecycle
  (start
    [this]
    (if (:executor-service this)
      this
      (assoc this :executor-service (make-executor-service-fn this))))
  (stop
    [this]
    (when-let [executor-service ^ExecutorService (:executor-service this)]
      (.shutdown executor-service)
      (if-let [termination-wait-ns (some-> termination-wait-duration Duration/.toNanos)]
        (when (pos? termination-wait-ns)
          (.awaitTermination executor-service termination-wait-ns TimeUnit/NANOSECONDS))
        (while (not (.isTerminated executor-service)))))
    (dissoc this :executor-service)))

(defn- adapt-rejected-execution-handler
  ^RejectedExecutionHandler [rejected-execution-handler]
  (cond
    (instance? RejectedExecutionHandler rejected-execution-handler)
    rejected-execution-handler
    (fn? rejected-execution-handler)
    (reify
      RejectedExecutionHandler
      (rejectedExecution
        [_this runnable executor]
        (rejected-execution-handler runnable executor)))
    :else
    (case (or rejected-execution-handler :abort-policy)
      :abort-policy (ThreadPoolExecutor$AbortPolicy.)
      :caller-runs-policy (ThreadPoolExecutor$CallerRunsPolicy.)
      :discard-oldest-policy (ThreadPoolExecutor$DiscardOldestPolicy.)
      :discard-policy (ThreadPoolExecutor$DiscardPolicy.))))

(defn make-thread-pool
  "Make a ThreadPoolExecutor for wrapping by ExecutorServiceComponent.

  - **core-pool-size** — number of threads to keep in the pool, even if they are idle
  - **max-pool-size** — maximum number of threads to allow in the pool
  - **keep-alive-duration** — when the number of threads is greater than the core, the maximum
    *java.time.Duration* to wait for new tasks before terminating
  - **work-queue** — queue to use for holding tasks before they are executed
  - **thread-factory** — factory to use when executor creates a new thread.  Defaults to
    *Executors/defaultThreadFactory*.
  - **rejected-execution-handler** — handler to use when execution is blocked because the thread
    bound and queue capacities are reached

  See *java.util.concurrent.ThreadPoolExecutor*"
  [{:keys [core-pool-size max-pool-size keep-alive-duration work-queue thread-factory
           rejected-execution-handler]}]
  (let [keep-alive-ns (-> keep-alive-duration Duration/.toNanos)]
    (ThreadPoolExecutor. core-pool-size max-pool-size keep-alive-ns TimeUnit/NANOSECONDS
      work-queue (or thread-factory (Executors/defaultThreadFactory))
      (-> rejected-execution-handler adapt-rejected-execution-handler))))

(defn thread-pool
  "Make a new ExecutorServiceComponent wrapping a ThreadPoolExecutor.

  - **core-pool-size** — number of threads to keep in the pool, even if they are idle
  - **max-pool-size** — maximum number of threads to allow in the pool
  - **keep-alive-duration** — when the number of threads is greater than the core, the maximum
    *java.time.Duration* to wait for new tasks before terminating
  - **work-queue** — queue to use for holding tasks before they are executed
  - **thread-factory** — factory to use when executor creates a new thread.
  - **convey-bindings?** — if true then convey thread local var bindings with tasks as they are
    submitted to the wrapped ExecutorService.
  - **rejected-execution-handler** — handler to use when execution is blocked because the thread
    bound and queue capacities are reached
  - **termination-wait-duration** — a *java.time.Duration* to wait for the ExecutorService to
    terminate when the component is stopped.  A duration of zero means don't wait at all.  If not
    specified, then wait indefinitely."
  {:arglists '([core-pool-size max-pool-size keep-alive-duration work-queue
                & {:as opts :keys [thread-factory rejected-execution-handler convey-bindings?
                                   termination-wait-duration]}])}
  ^ExecutorService [core-pool-size max-pool-size keep-alive-duration work-queue & {:as opts}]
  (map->ExecutorServiceComponent
    :core-pool-size core-pool-size
    :max-pool-size max-pool-size
    :keep-alive-duration keep-alive-duration
    :work-queue work-queue
    :make-executor-service-fn make-thread-pool
    opts))

(defn- scheduled-executor-service!
  ^ScheduledExecutorService [{:keys [^ScheduledExecutorService executor-service]}]
  (when-not executor-service
    (throw (IllegalStateException. "ScheduledExecutorServiceComponent is not started")))
  executor-service)

(desiderata/defrecord ScheduledExecutorServiceComponent
  "An scheduled executor service that is also a component.  It can be injected as a dependency of
  other components, but also implements the ScheduledExecutorService interface.

  - **make-executor-service-fn** — a function that takes the ScheduledExecutorServiceComponent as
    an argument and creates a ScheduledExecutorService for wrapping.  Any options necessary for
    constructing an ScheduledExecutorService should be taken from the
    ScheduledExecutorServiceComponent.
  - **convey-bindings?** — if true then convey thread local var bindings with tasks as they are
    submitted to the wrapped ExecutorService.
  - **termination-wait-duration** — a *java.time.Duration* to wait for the ScheduledExecutorService
    to terminate when this component is stopped.  A duration of zero means don't wait at all.  If
    not specified, then wait indefinitely."
  [make-executor-service-fn termination-wait-duration]
  ScheduledExecutorService
  (awaitTermination
    [this timeout unit]
    (.awaitTermination (scheduled-executor-service! this) timeout unit))
  (invokeAll
    [this tasks]
    (.invokeAll (scheduled-executor-service! this) (bind-all this tasks)))
  (invokeAll
    [this tasks timeout unit]
    (.invokeAll (scheduled-executor-service! this) (bind-all this tasks) timeout unit))
  (invokeAny
    [this tasks]
    (.invokeAny (scheduled-executor-service! this) (bind-all this tasks)))
  (invokeAny
    [this tasks timeout unit]
    (.invokeAny (scheduled-executor-service! this) (bind-all this tasks) timeout unit))
  (isShutdown
    [this]
    (.isShutdown (scheduled-executor-service! this)))
  (isTerminated
    [this]
    (.isTerminated (scheduled-executor-service! this)))
  (shutdown
    [this]
    (.shutdown (scheduled-executor-service! this)))
  (shutdownNow
    [this]
    (.shutdownNow (scheduled-executor-service! this)))
  (^Future submit
   [this ^Runnable task]
   (.submit (scheduled-executor-service! this) (bind-runnable this task)))
  (^Future submit
   [this ^Runnable task result]
   (.submit (scheduled-executor-service! this) (bind-runnable this task) result))
  (^Future submit
   [this ^Callable task]
   (.submit (scheduled-executor-service! this) (bind-callable this task)))
  (execute
    [this command]
    (.execute (scheduled-executor-service! this) (bind-runnable this command)))
  (^ScheduledFuture schedule
   [this ^Runnable command ^long delay ^TimeUnit unit]
   (.schedule (scheduled-executor-service! this) (bind-runnable this command) delay unit))
  (^ScheduledFuture schedule
   [this ^Callable command ^long delay ^TimeUnit unit]
   (.schedule (scheduled-executor-service! this) (bind-callable this command) delay unit))
  (scheduleAtFixedRate
    [this command initial-delay period unit]
    (.scheduleAtFixedRate (scheduled-executor-service! this) (bind-runnable this command)
      initial-delay period unit))
  (scheduleWithFixedDelay
    [this command initial-delay delay unit]
    (.scheduleAtFixedRate (scheduled-executor-service! this) (bind-runnable this command)
      initial-delay delay unit))
  component/Lifecycle
  (start
    [this]
    (if (:executor-service this)
      this
      (assoc this :executor-service (make-executor-service-fn this))))
  (stop
    [this]
    (when-let [executor-service ^ExecutorService (:executor-service this)]
      (.shutdown executor-service)
      (if-let [termination-wait-ns (some-> termination-wait-duration Duration/.toNanos)]
        (when (pos? termination-wait-ns)
          (.awaitTermination executor-service termination-wait-ns TimeUnit/NANOSECONDS))
        (while (not (.isTerminated executor-service)))))
    (dissoc this :executor-service)))

(defn make-scheduled-thread-pool
  "Make a ScheduledThreadPoolExecutor for wrapping by ScheduledExecutorServiceComponent.

  - **core-pool-size** — number of threads to keep in the pool, even if they are idle
  - **thread-factory** — factory to use when executor creates a new thread.  Defaults to
    *Executors/defaultThreadFactory*.
  - **rejected-execution-handler** — handler to use when execution is blocked because the thread
    bound and queue capacities are reached

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  [{:keys [core-pool-size thread-factory rejected-execution-handler]}]
  (ScheduledThreadPoolExecutor. core-pool-size (or thread-factory (Executors/defaultThreadFactory))
    (-> rejected-execution-handler adapt-rejected-execution-handler)))

(defn scheduled-thread-pool
  "Make a new ScheduledExecutorServiceComponent wrapping a ScheduledThreadPoolExecutor.

  - **core-pool-size** — number of threads to keep in the pool, even if they are idle
  - **thread-factory** — factory to use when executor creates a new thread.  Defaults to
    *Executors/defaultThreadFactory*.
  - **convey-bindings?** — if true then convey thread local var bindings with tasks as they are
    submitted to the wrapped ScheduledExecutorService.
  - **rejected-execution-handler** — handler to use when execution is blocked because the thread
    bound and queue capacities are reached
  - **termination-wait-duration** — a *java.time.Duration* to wait for the ScheduledExecutorService
    to terminate when the component is stopped.  A duration of zero means don't wait at all.  If not
    specified, then wait indefinitely."
  {:arglists '([core-pool-size
                & {:as opts :keys [thread-factory rejected-execution-handler
                                   convey-bindings? termination-wait-duration]}])}
  ^ScheduledExecutorService [core-pool-size & {:as opts}]
  (map->ScheduledExecutorServiceComponent
    :core-pool-size core-pool-size
    :make-executor-service-fn make-scheduled-thread-pool
    opts))

(defn- scheduled-future!
  ^ScheduledFuture [{:keys [scheduled-future]}]
  (when-not scheduled-future
    (throw (Exception. "ScheduledTaskComponent is not started")))
  scheduled-future)

(desiderata/defrecord ScheduledTaskComponent
  "A scheduled task that is also a component.

  - **executor-service** — ScheduledExecutorService to use to schedule task.  If not specified then
    a single thread `scheduled-thread-pool` component is started when this component is started and
    stopped when this component is stopped.
  - **make-scheduled-future-fn** — a function that takes the ScheduledTaskComponent as an argument
    and creates a ScheduledFuture for wrapping.  Any options necessary for constructing a
    ScheduledFuture should be taken from the ScheduledTaskComponent."
  [^ExecutorService executor-service make-scheduled-future-fn]
  ScheduledFuture
  (compareTo
    [this other]
    (.compareTo (scheduled-future! this) other))
  (getDelay
    [this unit]
    (.getDelay (scheduled-future! this) unit))
  (cancel
    [this may-interrupt-if-running?]
    (.cancel (scheduled-future! this) may-interrupt-if-running?))
  (get
    [this]
    (.get (scheduled-future! this)))
  (get
    [this timeout unit]
    (.get (scheduled-future! this) timeout unit))
  (isCancelled
    [this]
    (.isCancelled (scheduled-future! this)))
  (isDone
    [this]
    (.isDone (scheduled-future! this)))
  component/Lifecycle
  (start
    [this]
    (if (:scheduled-future this)
      this
      (let [stop-executor-service? (not executor-service)
            executor-service (or executor-service (component/start (scheduled-thread-pool 1)))
            this (assoc this
                   :executor-service executor-service
                   :stop-executor-service? stop-executor-service?)]
        (assoc this :scheduled-future (make-scheduled-future-fn executor-service this)))))
  (stop
    [{:as this :keys [^ScheduledFuture scheduled-future stop-executor-service?]}]
    (when-let [scheduled-future scheduled-future]
      (.cancel scheduled-future false))
    (when stop-executor-service?
      (component/stop executor-service))
    (cond-> (dissoc this :scheduled-future :stop-executor-service?)
      stop-executor-service? (assoc :executor-service nil))))

(defn make-fixed-rate-scheduled-task
  "Make a ScheduledFuture for wrapping by ScheduledTaskComponent.

  - **executor-service** — ScheduledExecutorService to use to schedule task.  If not specified then
    a single thread `scheduled-thread-pool` component is started when this component is started and
    stopped when this component is stopped.
  - **task** — a one argument function to run as a scheduled task.  It takes as an argument the
    task componentand its dependencies.
  - **initial-delay-duration** — *java.time.Duration* to wait before the first run of `task`
  - **period-duration** — *java.time.Duration* from the start of one run to the start of the next

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  [^ScheduledExecutorService executor-service
   {:as component :keys [task initial-delay-duration period-duration]}]
  (let [initial-delay-ns (or (some-> initial-delay-duration Duration/.toNanos) 0)
        period-ns (some-> period-duration Duration/.toNanos)]
    (.scheduleAtFixedRate executor-service #(task component) initial-delay-ns period-ns
      TimeUnit/NANOSECONDS)))

(defn fixed-rate-scheduled-task
  "Make a new fixed rate ScheduledTaskComponent wrapping a ScheduledFuture.

  - **task** — a one argument function to run as a scheduled task.  It takes as an argument the
    task component and its dependencies.
  - **period-duration** — *java.time.Duration* from the start of one run to the start of the next
  - **initial-delay-duration** — *java.time.Duration* to wait before the first run of `task`,
    defaults to 0
  - **executor-service** — ScheduledExecutorService to use to schedule task.  If not specified then
    a single thread `scheduled-thread-pool` component is started when this component is started and
    stopped when this component is stopped.

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  {:arglists '([task period-duration & {:as opts :keys [initial-delay-duration executor-service]}])}
  ^ScheduledFuture [task period-duration & {:as opts}]
  (map->ScheduledTaskComponent
    :task task
    :period-duration period-duration
    :make-scheduled-future-fn make-fixed-rate-scheduled-task
    opts))

(defn make-fixed-delay-scheduled-task
  "Make a ScheduledFuture for wrapping by ScheduledTaskComponent.

  - **executor-service** — ScheduledExecutorService to use to schedule task.  If not specified then
    a single thread `scheduled-thread-pool` component is started when this component is started and
    stopped when this component is stopped.
  - **task** — a one argument function to run as a scheduled task.  It takes as an argument the
    task component and its dependencies.
  - **initial-delay-duration** — *java.time.Duration* to wait before the first run of `task`
  - **delay-duration** — *java.time.Duration* from the end of one run to the start of the next

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  [^ScheduledExecutorService executor-service
   {:as component :keys [task initial-delay-duration delay-duration]}]
  (let [initial-delay-ns (or (some-> initial-delay-duration Duration/.toNanos) 0)
        delay-ns (some-> delay-duration Duration/.toNanos)]
    (.scheduleWithFixedDelay executor-service #(task component) initial-delay-ns delay-ns
      TimeUnit/NANOSECONDS)))

(defn fixed-delay-scheduled-task
  "Make a new fixed delay ScheduledTaskComponent wrapping a ScheduledFuture.

  - **task** — a one argument function to run as a scheduled task.  It takes as an argument the
    task component and its dependencies.
  - **delay-duration** — *java.time.Duration* from the end of one run to the start of the next
  - **initial-delay-duration** — *java.time.Duration* to wait before the first run of `task`,
    defaults to 0
  - **executor-service** — ScheduledExecutorService to use to schedule task.  If not specified then
    a single thread `scheduled-thread-pool` component is started when this component is started and
    stopped when this component is stopped.

  See *java.util.concurrent.ScheduledThreadPoolExecutor*"
  {:arglists '([task delay-duration & {:as opts :keys [initial-delay-duration executor-service]}])}
  ^ScheduledFuture [task delay-duration & {:as opts}]
  (map->ScheduledTaskComponent
    :task task
    :delay-duration delay-duration
    :make-scheduled-future-fn make-fixed-delay-scheduled-task
    opts))
