;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.amalgam
  (:refer-clojure :exclude [vector])
  (:require
    [com.stuartsierra.component :as component]))

(defn overwrite
  "Resolve current and configuration values by always taking the configuration value."
  [_v c]
  c)

(defn configure
  "Recursively merge config map into component using merge-fns as necessary.  Maps are recursively
  merged.  Sets, vectors and lists are non-recursively concatenated.

  This behavior can be overridden with merge-fns, which should match the structure of config and
  value.  A merge function takes two arguments: the current value, and the configured value
  (respectively) and should return the final, resolved value.

  Example:

  ```clojure
  user> (amalgam/configure {:a [1] :b {:c [2]}}
          {:a [10] :b {:c [20] :d 30}}
          {:b {:c amalgam/overwrite}})
  {:a [1 10], :b {:c [20], :d 30}}
  ```"
  ([component config] (configure component config {}))
  ([component config merge-fn]
   (if (fn? merge-fn)
     (merge-fn component config)
     (cond
       (and (map? config) (map? component))
       (reduce-kv
         (fn [m k c]
           (assoc m k
             (if (contains? m k)
               (configure (get m k) c (when (and (map? merge-fn) (contains? merge-fn k))
                                        (get merge-fn k)))
               c)))
         component
         config)
       (or (and (set? config) (set? component))
         (and (vector? config) (vector? component)))
       (into component config)
       (and (list? config) (list? component))
       (apply list (concat component config))
       :else
       config))))

(defn start-system
  "Construct, configure, then start a system.

  - **`make-system`** — no argument function returning a new system
  - **`read-config`** — no argument function returning the configuration map.  The configuration map
    should mirror the structure of the system.
  - **`merge-fns`** (optional) — map mirroring the structure of both the system and configuration
    maps that contains merge functions to resolve a final value from the current value and
    configured value.

  Example:

  ```clojure
  user> (amalgam/start-system #(component/system-map :component (map->Component {:foo :bar}))
          (constantly {:component {:foo :baz}})
          {:component {:foo amalgam/overwrite}})
  {:component {:foo :baz}}
  ```

  See [[configure]]"
  ([make-system read-config]
   (start-system make-system read-config {}))
  ([make-system read-config merge-fns]
   (-> (make-system)
     (configure (read-config) merge-fns)
     component/start)))

(defn run-system
  "Start a configured system, run it, and register a JVM shutdown hook to stop the system.  Blocks
  until the system stop.

  - **`make-system`** — no argument function returning a new system
  - **`read-config`** — no argument function returning the configuration map.  The configuration map
    should mirror the structure of the system.
  - **`merge-fns`** (optional) — map mirroring the structure of both the system and configuration
    maps that contains merge functions to resolve a final value from the current value and
    configured value.

  See [[configure]] [[start-system]]"
  ([make-system read-config]
   (run-system make-system read-config {}))
  ([make-system read-config merge-fns]
   (let [running? (promise)
         system (start-system make-system read-config merge-fns)]
     (.addShutdownHook (Runtime/getRuntime)
       (Thread.
         #(do (component/stop system)
            (deliver running? false))))
     (while (try @running? (catch InterruptedException _ false))))))

(defn component
  "Make a component that from an object by extending component/Lifecycle via metadata.

  The object must take matadata, and generally it is best as a map so it can be
  configured (see [[configure]]).  The started component (returned from `start`) must also take
  metadata.

  The start and stop functions are optional, and if they are not given, then they default to
  no-ops.

  If a type is given, it will be set as the object's type (i.e. the `type` key of its metadata).

  After this component has been started, stopping it resets it back to `object`.  In other words,

  ```clojure
  (let [c (amalgam/component ...)]
    (assert (identical? c (component/stop (component/start c)))))
  ```

  - **`start`** (optional) — a single argument function taking the object and returning a started
    component, which must be able to take metadata.
  - **`stop`** (optional) — a single argument function taking the started component and releasing
    resources.  `stop` is side-effecting and its return value is ignored.
  - **`type`** (optional) — a symbol to set as the objects type (i.e. in the `:type` key of its
    metadata).

  Example:

  ```clojure
  (defn temp-file
    [& {:as opts :keys [path]}]
    (amalgam/component opts
      :start (fn [this] (assoc this :file (io/file (:path this))))
      :stop (fn [this] (io/delete-file (:file this) true))))
  (def my-system (component/system-map :scratch-file (temp-file :path \"/tmp/scratch\")))
  ```"
  [object & {:keys [start stop type]}]
  (cond-> (vary-meta object
            assoc
            `component/start
            (fn [this]
              (vary-meta (cond-> this start start)
                assoc
                `component/start
                (fn [this'] this')
                `component/stop
                (fn [this']
                  (when stop (stop this'))
                  this))))
    type (vary-meta assoc :type type)))

(defn ^:deprecated make-component-fn
  "Make a component constructor that takes options as keyword args and returns a component.

  After this component has been started, stopping it resets it back to the options map.  In other
  words,

  ```clojure
  (let [make-component (make-component-fn ...)
        c (make-component ...)]
    (assert (identical? c (component/stop (component/start c)))))
  ```

  - **`start`** — (optional) a single argument function taking the options and returning a new
    instance of the component.  The return value of `start` should be able to take metadata.
  - **`stop`** — (optional) a single argument function taking the component and releasing resources.
    `stop` is side-effecting and its return value is ignored.

  Example:

  ```clojure
  (def temp-file
    (amalgam/make-component-fn
      :start (fn [this] (assoc this :file (io/file (:path this))))
      :stop (fn [this] (io/delete-file (:file this) true))))
  (def my-system (component/system-map :scratch-file (temp-file :path \"/tmp/scratch\")))
  ```"
  [& {:keys [start stop]}]
  (fn [& {:as opts}]
    (vary-meta opts
      assoc
      `component/start
      (fn [this]
        (vary-meta (cond-> this start start)
          assoc
          `component/start
          (fn [this'] this')
          `component/stop
          (fn [this']
            (when stop (stop this'))
            this))))))

(defn vector
  "A vector component that collects its dependencies into a vector.

  Example:

  ```clojure
  user> (-> (component/system-map :v (amalgam/vector :a :b) :a 1 :b 2) component/start :v)
  [1 2]
  ```"
  [& dependency-keys]
  (let [dependency-keys (vec dependency-keys)]
    (component/using
      (component {::dependency-keys (vec dependency-keys)}
        :type `VectorComponent
        :start (fn [{:as this ::keys [dependency-keys]}]
                 (into [] (map this) dependency-keys)))
      dependency-keys)))
