(ns systems.thoughtfull.build
  (:require
    [clojure.java.io :as io]
    [clojure.tools.build.api :as b]
    [deps-deploy.deps-deploy :as dd]))

(def build-config
  (try (-> "build.edn"
         io/file
         io/reader
         java.io.PushbackReader.
         read)
    (catch Exception _)))

(def lib
  (:lib build-config))

(def lib-name
  (some-> lib name))

(def repo-name
  (or (some-> (:repo-name build-config) name) lib-name))

(def repo
  (str "thoughtfull-clojure/" repo-name))

(def version
  (:version build-config))

(def description
  (:description build-config))

(def jar-file
  (format "target/%s-%s.jar" lib-name version))

(def class-dir
  "target/classes")

(defn validate-attrs!
  []
  (assert (qualified-symbol? lib))
  (assert (= "systems.thoughtfull" (namespace lib)))
  (assert (re-matches #"\d+\.\d+\.\d+(?:-\w+)?" version))
  (assert (string? description))
  (assert (pos? (count description)))
  (assert (string? repo-name))
  (assert (pos? (count repo-name))))

(def pom-data
  [[:description description]
   [:url (str "https://github.com/" repo)]
   [:licenses
    [:license
     [:name "Mozilla Public License Version 2.0"]
     [:url "http://mozilla.org/MPL/2.0/"]]]
   [:developers
    [:developer
     [:name "technosophist"]]]
   [:scm
    [:url (str "https://github.com/" repo)]
    [:connection (str "scm:git:https://github.com/" repo ".git")]
    [:developerConnection (str "scm:git:ssh:git@github.com:" repo ".git")]
    [:tag (str lib-name "-" version)]]])

(def clojars-config
  (try
    (-> (str (System/getenv "HOME") "/.clojars.edn")
      io/file
      io/reader
      java.io.PushbackReader.
      read
      (select-keys [:username :password]))
    (catch Exception _)))

(def username
  (:username clojars-config))

(def password
  (:password clojars-config))

(defn clean
  [& _]
  (b/delete {:path "target"}))

(defn jar
  [& _]
  (validate-attrs!)
  (clean)
  (b/write-pom {:class-dir class-dir
                :lib lib
                :version version
                :basis (b/create-basis {:project "deps.edn"})
                :src-dirs ["src"]
                :pom-data pom-data
                :resource-dirs ["resources"]})
  (b/copy-dir {:src-dirs ["src" "resources"]
               :target-dir class-dir})
  (b/jar {:class-dir class-dir
          :jar-file jar-file}))

(defn deploy
  [& _]
  (validate-attrs!)
  (assert username)
  (assert password)
  (jar)
  (dd/deploy {:installer :remote
              :artifact (b/resolve-path jar-file)
              :pom-file (b/pom-path {:lib lib :class-dir class-dir})
              :repository {"clojars" {:url "https://clojars.org/repo"
                                      :username username
                                      :password password}}}))

(defn install
  [& _]
  (validate-attrs!)
  (jar)
  (dd/deploy {:installer :local
              :artifact (b/resolve-path jar-file)
              :pom-file (b/pom-path {:lib lib :class-dir class-dir})}))
