(ns systems.thoughtfull.desiderata
  (:refer-clojure :exclude [defrecord]))

(defn- parse-opts
  [opts+specs]
  (loop [[k v & s :as specs] opts+specs
         defaults nil
         opts []]
    (if (and (seq specs) (keyword? k))
      (if (= k ::defaults)
        (recur s v opts)
        (recur s defaults (conj opts k v)))
      [defaults opts specs])))

(defn- parse-initializer
  [name specs]
  (loop [[s & ss :as specs] specs
         not-initializer []]
    (if (seq specs)
      (if (and (list? s) (= (first s) name))
        [s (into not-initializer ss)]
        (recur ss (conj not-initializer s)))
      [nil not-initializer])))

(defmacro defrecord
  "Drop-in replacement for clojure.core/defrecord with extra functionality.  See
  clojure.core/defrecord for details about core functionality.

  As a relatively minor addition the factory function takes keyword arguments.  There are three
  other additions: a docstring, default values, and an initializer.

  A docstring, if given, is appended to the docstring for both the factory and positional factory
  functions.

  The :systems.thoughtfull.desiderata/defaults option can be given with a hash map to supply
  defaults.  The hash map supplies default values for the declared fields (and extra non-field
  keys and values).  Any values given as arguments to the factory or positional factory functions
  override these defaults. Example:

  (desiderata/defrecord Gizmo
    [name]
    ::desiderata/defaults {:name \"Gizmo\" :color :blue})

  If a method with the same name as the defrecord is defined, it is used to as an initializer.
  After the record is constructed by the factory or positional factory function, it is given to
  the initializer and the result is returned from the factory or positional factory function. If
  the initializer does not return an instance of the defrecord, an IllegalStateException is
  thrown.  Example:

  (desiderata/defrecord Company
    [debt equity]
    (Company
      [this]
      (assoc this :gearing-ratio (/ debt equity))))"
  {:arglists '([name docstring? [& fields] & opts+specs])}
  [name docstring? & [fields & opts+specs]]
  (let [[docstring fields opts+specs] (if (string? docstring?)
                                        [docstring? fields opts+specs]
                                        [nil docstring? (cons fields opts+specs)])
        [defaults opts specs] (parse-opts opts+specs)
        [initializer specs] (parse-initializer name specs)
        [init-params & init-body] (or (next initializer) `[[this#] this#])
        pfactory (symbol (format "->%s" name))
        factory (symbol (format "map->%s" name))]
    `(let [defaults# (fn [] ~defaults)
           record# (clojure.core/defrecord ~name ~fields ~@opts ~@specs)
           pfactory# ~pfactory
           factory# ~factory
           initializer# (fn [this#]
                          (let [{:keys [~@fields]} this#
                                ~init-params [this#]
                                this# (do ~@init-body)]
                            (when-not (instance? record# this#)
                              (throw (java.lang.IllegalStateException.
                                       (str "Initializer should return instance of " '~name))))
                            this#))]
       (defn ~pfactory
         {:doc (str (:doc (meta (var ~pfactory)))
                 (when-let [docstring# ~docstring]
                   (format "\n\n  %s" docstring#)))}
         [~@fields]
         (let [defaults# (defaults#)
               extra-keys# (into #{} (remove ~(into #{} (map keyword) fields)) (keys defaults#))]
           (initializer# (merge (pfactory# ~@fields) (select-keys defaults# extra-keys#)))))
       (defn ~factory
         {:doc (str (:doc (meta (var ~factory)))
                 (when-let [docstring# ~docstring]
                   (format "\n\n  %s" docstring#)))
          :arglists '([& {:keys [~@fields]}])}
         [& {:as args#}]
         (initializer# (factory# (merge (defaults#) args#))))
       record#)))
