;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.desiderata
  (:refer-clojure :exclude [defrecord])
  (:import
    (java.util.concurrent ThreadFactory)))

(set! *warn-on-reflection* true)

(defn- parse-opts
  [opts+specs]
  (loop [[k v & s :as specs] opts+specs
         defaults nil
         opts []]
    (if (and (seq specs) (keyword? k))
      (if (= k ::defaults)
        (recur s v opts)
        (recur s defaults (conj opts k v)))
      [defaults opts specs])))

(defn- parse-initializer
  [name specs]
  (loop [[s & ss :as specs] specs
         not-initializer []]
    (if (seq specs)
      (if (and (list? s) (= (first s) name))
        [s (into not-initializer ss)]
        (recur ss (conj not-initializer s)))
      [nil not-initializer])))

(defmacro defrecord
  "Drop-in replacement for `clojure.core/defrecord` with extra functionality.  See
  `clojure.core/defrecord` for details about core functionality.

  As a relatively minor addition the factory function takes keyword arguments.  Another somewhat
  minor addition, metadata on the name symbol is propagated to the factory functions.  There are
  three other additions: a docstring, default values, and an initializer.

  A docstring, if given, is appended to the docstring for both the factory and positional factory
  functions.

  Example:

  ```clojure
  user> (desiderata/defrecord Widget
  \"A widget for frobbing gizmos.

  width and height are in metric.\"
  [width height])
  user.Widget

  user> (doc map->Widget)
  -------------------------
  user/map->Widget
  ([& {:keys [width height]}])
    Factory function for class user.Widget, taking a map of keywords to field values.

    A widget for frobbing gizmos.

    width and height are in metric.
  nil
  ```

  The `:systems.thoughtfull.desiderata/defaults` option can be given with a hash map to supply
  defaults.  The hash map supplies default values for the declared fields (and extra non-field
  keys and values).  Any values given as arguments to the factory or positional factory functions
  override these defaults.

  Example:

  ```clojure
  user> (desiderata/defrecord Gizmo
  [name]
  ::desiderata/defaults
  {:name \"Gizmo\"
   :color :blue})
  user.Gizmo

  user> (->Gizmo \"the Great\")
  {:name \"the Great\", :color :blue}

  user> (map->Gizmo :texture :bumpy)
  {:name \"Gizmo\", :color :blue, :texture :bumpy}
  ```

  If a method with the same name as the defrecord is defined, it is used to as an initializer.
  After the record is constructed by the factory or positional factory function, it is given to
  the initializer and the result is returned from the factory or positional factory function. If
  the initializer does not return an instance of the type, an *IllegalStateException* is thrown.

  Example:

  ```clojure
  user> (desiderata/defrecord Company
  [debt equity]
  (Company
    [this]
    (assoc this :gearing-ratio (/ debt equity))))
  user.Company

  user> (->Company 100 1000)
  {:debt 100, :equity 1000, :gearing-ratio 1/10}
  ```

  - **`name`** — name of the type
  - **`docstring`** (optional) — appended to the docstrings of the factory and positional factory
    functions
  - **`fields`** — names of the fields of the type
  - **`opts+specs`** (optional) — options, interfaces, and methods

  See `clojure.core/defrecord`"
  {:arglists '([name docstring? [& fields] & opts+specs])}
  [name & args]
  (let [[docstring & args] (cond->> args (not (string? (first args))) (cons nil))
        [fields & args] args
        [defaults opts specs] (parse-opts args)
        [initializer specs] (parse-initializer name specs)
        [init-params & init-body] (or (next initializer) `[[this#] this#])
        pfactory (symbol (format "->%s" name))
        factory (symbol (format "map->%s" name))]
    `(let [defaults# (fn [] ~defaults)
           record# (clojure.core/defrecord ~name ~fields ~@opts ~@specs)
           pfactory# ~pfactory
           factory# ~factory
           initializer# (fn [this#]
                          (let [{:keys [~@fields]} this#
                                ~init-params [this#]
                                this# (do ~@init-body)]
                            (when-not (instance? record# this#)
                              (throw (java.lang.IllegalStateException.
                                       (str "Initializer should return instance of " '~name))))
                            this#))]
       (defn ~(with-meta pfactory (meta name))
         {:doc (str (:doc (meta (var ~pfactory)))
                 (when-let [docstring# ~docstring]
                   (format "\n\n  %s" docstring#)))
          :meta (meta (var ~pfactory))}
         [~@fields]
         (let [defaults# (defaults#)
               extra-keys# (into #{} (remove ~(into #{} (map keyword) fields)) (keys defaults#))]
           (initializer# (merge (pfactory# ~@fields) (select-keys defaults# extra-keys#)))))
       (defn ~(with-meta factory (meta name))
         {:doc (str (:doc (meta (var ~factory)))
                 (when-let [docstring# ~docstring]
                   (format "\n\n  %s" docstring#)))
          :arglists '([& {:keys [~@fields]}])
          :meta (meta (var ~factory))}
         [& {:as args#}]
         (initializer# (factory# (merge (defaults#) args#))))
       record#)))

(defn uncaught-exception-handler
  "Adapt `uncaught-exception-handler-fn` to a *Thread.UncaughtExceptionHandler*.
  `uncaught-exception-handler-fn` should be a function of two arguments (thread and
  throwable). Returns nil if `uncaught-exception-handler-fn` is nil.

  - **`uncaught-exception-handler-fn`** — function of two arguments (thread and throwable) to
    adapt as a *Thread.UncaughtExceptionHandler*.  Returns nil if `uncaught-exception-handler-fn` is
    nil.

  See *Thread.UncaughtExceptionHandler*"
  ^Thread$UncaughtExceptionHandler [uncaught-exception-handler-fn]
  (when uncaught-exception-handler-fn
    (reify Thread$UncaughtExceptionHandler
      (uncaughtException
        [_ thread throwable]
        (uncaught-exception-handler-fn thread throwable)))))

(defn set-default-uncaught-exception-handler-fn!
  "Set default *Thread.UncaughtExceptionHandler* to `uncaught-exception-handler-fn` after adapting
  it using [[uncaught-exception-handler]].  `uncaught-exception-handler-fn` should be a function
  of two arguments (thread and throwable).  If `uncaught-exception-handler-fn` is nil, then the
  default uncaught exception handler is cleared.

  - **`uncaught-exception-handler-fn`** — function of two arguments (thread and throwable) to set
    as the default *Thread.UncaughtExceptionHandler* after adapting it using
    [[uncaught-exception-handler]].  Returns nil if `uncaught-exception-handler-fn` is nil.

  See *Thread/setDefaultUncaughtExceptionHandler*, *Thread.UncaughtExceptionHandler*"
  [uncaught-exception-handler-fn]
  (Thread/setDefaultUncaughtExceptionHandler
    (uncaught-exception-handler uncaught-exception-handler-fn)))

(defonce ^:private pool-count (atom 0))

(defn thread-factory
  "Create *java.util.concurrent.ThreadFactory*.

  - **`name`** (optional) — prefix for thread names, e.g., a `name` of `\"widget-pool\"` will create
    threads named `\"widget-pool-thread-N\"` where N increments each time a thread is created.  If
    `name` is not given, it defaults to `\"pool-M\"` where M is incremented for each new thread
    factory.
  - **`convey-bindings?`** (optional) — if true convey to created threads the bindings established
    when the thread factory was created, defaults to false.
  - **`priority`** (optional) — initial thread priority for created threads, defaults to
    *Thread/NORMAL_PRIORITY*.
  - **`daemon?`** (optional) — if true threads should be marked as daemon threads, defaults to
    false.
  - **`uncaught-exception-handler-fn`** (optional) — function to set as a
    *Thread.UncaughtExceptionHandler* after adapting with [[uncaught-exception-handler]].  Bindings
    established when the thread factory is created are conveyed to `uncaught-exception-handler-fn`.
  - **`inherit-inheritable-thread-locals?`** (optional) — if true, then new threads inherit initial
    values for *InheritableThreadLocal* from constructing thread, defaults to true.  See
    *java.util.Thread/new*.

  See *java.util.concurrent.ThreadFactory*, *Thread.UncaughtExceptionHandler*"
  {:arglists '([& {:keys [name convey-bindings? priority daemon? uncaught-exception-handler-fn
                          inherit-inheritable-thread-locals?]}])}
  ^ThreadFactory
  [& {:as options
      :keys [name convey-bindings? priority daemon? uncaught-exception-handler-fn
             inherit-inheritable-thread-locals?]}]
  (let [daemon? (boolean daemon?)
        inherit-inheritable-thread-locals? (if (contains? options
                                                 :inherit-inheritable-thread-locals?)
                                             (boolean inherit-inheritable-thread-locals?)
                                             true)
        prefix (format "%s-thread-" (or name (format "pool-%d" (swap! pool-count inc))))
        thread-count (atom 0)
        frame (clojure.lang.Var/cloneThreadBindingFrame)
        ueh (when uncaught-exception-handler-fn
              (uncaught-exception-handler
                (fn [thread throwable]
                  (clojure.lang.Var/resetThreadBindingFrame frame)
                  (uncaught-exception-handler-fn thread throwable))))]
    (reify ThreadFactory
      (^Thread newThread [_ ^Runnable r]
       (let [r (if convey-bindings?
                 (fn []
                   (clojure.lang.Var/resetThreadBindingFrame frame)
                   (.run r))
                 r)
             t (Thread. nil r (str prefix (swap! thread-count inc)) 0
                 inherit-inheritable-thread-locals?)]
         (.setPriority t (or priority Thread/NORM_PRIORITY))
         (when ueh (.setUncaughtExceptionHandler t ueh))
         (.setDaemon t daemon?)
         t)))))
