;; Copyright © technosophist
;;
;; This Source Code Form is subject to the terms of the Mozilla Public License, v. 2.0. If a copy of
;; the MPL was not distributed with this file, You can obtain one at http://mozilla.org/MPL/2.0/.
;;
;; This Source Code Form is "Incompatible With Secondary Licenses", as defined by the Mozilla Public
;; License, v. 2.0.
(ns systems.thoughtfull.transductio
  (:require
    [clojure.java.io :as io])
  (:import
    (java.io OutputStream Writer)
    (java.lang AutoCloseable)))

(set! *warn-on-reflection* true)

(defn decoder
  "Returns a function that takes a source and returns a reducible that, opens the source, repeatedly
  decodes data from it, and closes it when it is done.

  open takes the source and returns a stateful, mutable object that will be given to decode and
  eventually to close.

  decode is given the opened source and must return either a decoded value or a sentinel
  indicating there is no more data.

  done? is given each decoded value and if it returns truthy, then the decoding ends.

  close must not be nil.  If close is not given, then it defaults to
  java.lang.AutoCloseable/.close.

  When the reduction completes whether normally, early (someone returns a reduced), or because of
  an exception, then close is called.

  See stream-decoder and reader-decoder."
  ([open decode done?]
   (decoder open decode done? nil))
  ([open decode done? close]
   (fn [source]
     (let [close (or close AutoCloseable/.close)]
       (reify
         clojure.lang.IReduceInit
         (reduce
           [_ rf init]
           (let [in (open source)]
             (try
               (loop [acc init
                      v (decode in)]
                 (if (done? v)
                   acc
                   (let [acc (rf acc v)]
                     (if (reduced? acc)
                       @acc
                       (recur acc (decode in))))))
               (finally
                 (close in))))))))))

(defn stream-decoder
  "Returns a function that takes a source and returns a reducible that, opens the source as a
  java.io.InputStream, repeatedly decodes data from it, and closes it when it is done.

  The decoder opens the source with clojure.java.io/input-stream and opts.

  decode is given the opened source and must return either a decoded value or a sentinel
  indicating there is no more data.

  done? is given each decoded value and if it returns truthy, then the decoding ends.

  The decoder closes source with java.lang.AutoCloseable/.close.

  See decoder and reader-decoder."
  [decode done?]
  (decoder io/input-stream decode done?))

(defn reader-decoder
  "Returns a function that takes a source and returns a reducible that, opens the source as a
  java.io.Reader, repeatedly decodes data from it, and closes it when it is done.

  The decoder opens the source with clojure.java.io/reader and opts.

  decode is given the opened source and must return either a decoded value or a sentinel
  indicating there is no more data.

  done? is given each decoded value and if it returns truthy, then the decoding ends.

  The decoder closes source with java.lang.AutoCloseable/.close.

  See decoder and stream-decoder."
  ([decode done?]
   (decoder io/reader decode done?))
  ([decode done? encoding]
   (decoder #(io/reader % :encoding encoding) decode done?)))

(defn encoder
  "Returns a function that takes either a sink and a source, reducing the source into the sink, or a
  sink, a transducer, and a source, transducing the source into the sink through the transducer.
  This is an analog of into but for a mutable sink.

  open takes the sink and returns a stateful, mutable object that will be given to encode! and
  eventually to close.

  encode! is given the opened sink and a value and must mutate the sink with the value.  The
  return value from encode! is ignored.

  close must not be nil.  If close is not given, then it defaults to
  java.lang.AutoCloseable/.close.

  When the reduction/transduction completes whether normally, early (someone returns a reduced),
  or because of an exception, then close is called.

  See stream-encoder and writer-encoder"
  ([open encode!]
   (encoder open encode! nil))
  ([open encode! close]
   (fn
     ([sink source]
      (let [out (open sink)
            close (or close AutoCloseable/.close)]
        (try
          (reduce (fn [out value] (encode! out value) out) out source)
          nil
          (finally
            (close out)))))
     ([sink xf source]
      (let [out (open sink)
            close (or close AutoCloseable/.close)]
        (try
          (transduce xf (fn ([out] out) ([out value] (encode! out value) out)) out source)
          nil
          (finally
            (close out))))))))

(defn stream-encoder
  "Returns a function that takes either a sink and a source, reducing the source into the sink, or a
  sink, a transducer, and a source, transducing the source into the sink through the transducer.
  This is an analog of into but for a java.io.OutputStream.

  The encoder opens the sink with clojure.java.io/output-stream and opts.

  The source (or the transducer applied to the source) must return byte arrays which are written
  to the output stream.

  When the reduction/transduction completes whether normally, early (someone returns a reduced),
  or because of an exception, then the stream is closed.

  See encoder and writer-encoder"
  ([]
   (encoder io/output-stream ^[bytes] OutputStream/.write))
  ([append]
   (encoder #(io/output-stream % :append append) ^[bytes] OutputStream/.write)))

(defn writer-encoder
  "Returns a function that takes either a sink and a source, reducing the source into the sink, or a
  sink, a transducer, and a source, transducing the source into the sink through the transducer.
  This is an analog of into but for a java.io.Writer.

  The encoder opens the sink with clojure.java.io/writer and opts.

  The source (or the transducer applied to the source) must return Strings which are written to
  the output stream.

  When the reduction/transduction completes whether normally, early (someone returns a reduced),
  or because of an exception, then the stream is closed.

  See encoder and stream-encoder"
  ([]
   (encoder io/writer ^[String] Writer/.write))
  ([append]
   (encoder #(io/writer % :append append) ^[String] Writer/.write)))
