; Copyright (c) Sławek Gwizdowski
;
; Permission is hereby granted, free of charge, to any person obtaining
; a copy of this software and associated documentation files (the "Software"),
; to deal in the Software without restriction, including without limitation
; the rights to use, copy, modify, merge, publish, distribute, sublicense,
; and/or sell copies of the Software, and to permit persons to whom the
; Software is furnished to do so, subject to the following conditions:
;
; The above copyright notice and this permission notice shall be included
; in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
; OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
; THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
; IN THE SOFTWARE.
;
(ns ^{:author "Sławek Gwizdowski"
      :doc "Essbase transaction logs.

ALG file is just pairs of timestamps and transaction descriptions:

* First two lines is the time stamp of when audit log was enabled.
* Remaining pairs describe user and location+length in ATX file, line wise.

ATX file holds the data as it was locked and sent:

* Quoted member names, non-quoted values.
* Data chunks are separated by empty lines.

This namespace lets you process transaction logs, filter and pack results
in a presentable way. Contains some basic predicates to aid that.
"}
 szew.essbase.txl
 (:require
   [clojure.string :as string]
   [clojure.java.io :as clj.io :refer [reader]]
   [clj-time.format :as time.format]
   [clj-time.core :as time.core]
   [szew.io :as io]
   [clojure-csv.core :as csv])
 (:import
   [java.util Locale]
   [java.io BufferedReader]
   [clojure.lang IFn]))

(def ^{:doc "Input format for ALG timestamp."}
  header-time-date-in
  (time.format/with-locale
    (time.format/formatter "[EEE MMM dd HH:mm:ss yyyy]")
    Locale/US))

(def ^{:doc "Output format for ALG timestamp."}
  header-time-date-out
  (time.format/with-locale
    (time.format/formatters :date-hour-minute-second)
    Locale/US))

(def ^{:doc "Regular expression for ALG entries."}
  header-payload-re #"(?x)Log\ Updates\ From\ User\ \[\ (.+)\ \]\s
                          Starting\ At\ Row\ \[\ (\d+)\ \]\s
                          For\ A\ Total\ Of\ \[\ (\d+)\ \]\ Rows")

(defn ->header
  "Consumes header index and lines, returns a header.
  "
  [idx two-line]
  (try
    (let [[u i d] (rest (re-find header-payload-re (second two-line)))
          ts      (time.format/parse header-time-date-in (first two-line))]
      {:timestamp ts
       :date (time.format/unparse header-time-date-out ts)
       :user u
       :at (dec (Integer/parseInt i)) ;; alg index=1, in file index=0
       :rows (dec (Integer/parseInt d)) ;; empty row is counted
       :raw two-line
       :index idx})
    (catch Exception ex
      (throw (ex-info "Oops in ->header." {:index idx :lines two-line} ex)))))

(defn ->block
  "Consumes block index & indexed lines, returns a block.
  "
  [idx lines]
  {:block lines
   :index idx
   :rows (count lines)
   :at (first (first lines))})

(defn headers
  "Consume line-seq, returns sequence of headers (via ->header).
  "
  [a-line-seq]
  (sequence (comp (drop 1) (map-indexed ->header))
            (partition 2 2 "" a-line-seq)))

(defn blocks
  "Consumes line-seq, returns sequence of blocks (via ->block).
  "
  [a-line-seq]
  (sequence (comp (map-indexed vector)
                  (partition-by (comp (partial not= "") second))
                  (filter (comp (partial not= "") second first))
                  (map-indexed ->block))
            a-line-seq))

(defn header+block
  "Consumes header and block, merges the structure and adds control checks.
  "
  [header block]
  {:head header
   :data block
   :ctrl {:rows-ok? (= (:rows header) (:rows block))
          :at-ok? (= (:at header) (:at block))
          :at-drift (- (:at header) (:at block))
          :rows-drift (- (:rows header) (:rows block))}})

(defn strict-header+block
  "Executes header+block, then throws exception if any checks failed.
  "
  [header block]
  (let [h+b (header+block header block)]
    (when-not (get-in h+b [:ctrl :rows-ok?] false)
      (throw (ex-info "Rows =/= block size" {:header+block h+b})))
    (when-not (get-in h+b [:ctrl :at-ok?] false)
      (throw (ex-info "Index =/= block start" {:header+block h+b})))
    h+b))

(defrecord TxLog [strict processor]
  io/Input
  (io/in! [spec {:keys [alg atx] :as source}]
    (with-open [^BufferedReader l (reader alg :encoding "UTF-8")
                ^BufferedReader t (reader atx :encoding "UTF-8")]
      (if strict
        (processor (map strict-header+block
                        (headers (line-seq l))
                        (blocks (line-seq t))))
        (processor (map header+block
                        (headers (line-seq l))
                        (blocks (line-seq t)))))))
  IFn
  (applyTo [spec args]
    (when-not (= (count args) 1)
      (throw (ex-info "Wrong number of arguments! Expected 1."
                      {:spec spec
                       :args args})))
    (io/in! spec (first args)))
  (invoke [spec source]
    (io/in! spec source)))

(defn tx-log
  "Will create an instance of TxLog. This instance defined io/in! call:

  (io/in! tx-instance {:alg path-to-alg :atx path-to-atx})
  "
  ([spec]
   (into (tx-log) spec))
  ([]
   (TxLog. false vec)))

;; ## some helpers

(defn drifts
  "Processor for TxLog that will allow ALG vs ATX check (at and rows).
  "
  [inputs]
  (loop [roll {:at-ok? (sorted-map)
               :rows-ok? (sorted-map)
               :first-processed (get-in (first inputs) [:head :index])}
         hits inputs]
    (if-let [curr (first hits)]
      (let [{:keys [at-ok? rows-ok?]} (:ctrl curr)
            index (get-in curr [:head :index])
            last-at (get-in roll [:at-ok? (:last-at-idx roll)])
            last-rows (get-in roll [:rows-ok? (:last-rows-idx roll)])]
        (recur (merge roll
                      {:last-processed index}
                      (when-not (= last-at at-ok?)
                        {:at-ok? (assoc (:at-ok? roll) index at-ok?)
                         :last-at-idx index})
                      (when-not (= last-rows rows-ok?)
                        {:rows-ok? (assoc (:rows-ok? roll) index rows-ok?)
                         :last-rows-idx index}))
               (rest hits)))
      roll)))

(defn between?
  "Returns predicate that checks :head :timestamp against given interval.

  Format required is clj-time/formatters :date-time - 2017-05-24T21:08:28.458Z
  "
  [date-time-from date-time-to]
  (let [fmt (time.format/formatters :date-time)
        from (time.format/parse fmt date-time-from)
        to (time.format/parse fmt date-time-to)
        interval (time.core/interval from to)]
    (fn between-checker [entry]
      (let [t (-> entry :head :timestamp)]
        (time.core/within? interval t)))))

(defn user-matches?
  "Returns predicate that checks :head :user against given regexp.
  "
  [re-term]
  (fn user-match-checker [entry]
    (let [d (-> entry :head :user)]
      (try
        (re-find re-term d)
        (catch Exception ex (throw (ex-info "ops" {:entry entry})))))))

(defn data-matches?
  "Returns predicate that checks :data :block with given regexp.
  "
  [re-term]
  (fn data-match-checker [entry]
    (let [d (-> entry :data :block)]
      (try
        (some (comp (partial re-find re-term) second) d)
        (catch Exception ex (throw (ex-info "ops" {:entry entry})))))))

(defn printable
  "Packs :head :raw and :data :block into ALG+ATX form.

  Header occupies first column, data is shifted one column to the right.
  It can be cleanly dumped as CSV/TSV and imported into a spreadsheet.
  "
  [picks]
  (letfn [(header [entry]
            (->> entry :head :raw (mapv vector)))
          (shift [lines]
            (mapv (partial into [""]) lines))
          (body [entry]
            (let [lines (mapv second (-> entry :data :block))
                  lump (string/join "\n" lines)
                  rows (csv/parse-csv lump :delimiter \space :strict true)]
              (into (shift rows) [[""] [""]])))
          (pack-it [entry]
            (into (header entry) (body entry)))]
    (mapcat pack-it picks)))
