# szew/essbase

Consume various files produced by Essbase.

[![szew/essbase](https://clojars.org/szew/essbase/latest-version.svg)](https://clojars.org/szew/essbase)

[API documentation][latest] | [Changelog][changelog]

_Please note: API breakage will happen in minor versions (0.X.0) before 1.0.0
is released. Only patch versions (0.n.X) are safe until then._

** Table of Contents **

[TOC]

## Why

I've been dogfooding my private Clo*j*ure toolbox (named `szew`) since 2012.

Splitting out and releasing non-proprietary parts.

### What's here

Namespace breakdown:

* `szew.essbase.aso`  -- ASO data export
* `szew.essbase.bso`  -- BSO data export
* `szew.essbase.cols` -- Column export
* `szew.essbase.logs` -- Application logs and MaxL spools
* `szew.essbase.otl`  -- Outline XML export
* `szew.essbase.txl`  -- BSO transaction logs (alg+atx)

Processing is based on contracts established by `szew.io`, namely `szew.io/in!`
protocol method, but most of the API should hide that fact until you need
to go deeper than `predicate?` + data file.

## Usage

Like all the other `szew` pieces this library is REPL workflow oriented and
with a bit of scripting in mind. Please forgive scoped `def`s below, these
are here for presentation purposes only.

```clojure
(ns dev
 (:require
   [eftest.runner :refer [find-tests run-tests]]
   [szew.io :as io]
   [szew.essbase.aso :as ess.aso]
   [szew.essbase.bso :as ess.bso]
   [szew.essbase.cols :as ess.cols]
   [szew.essbase.logs :as ess.logs]
   [szew.essbase.txl :as ess.txl]
   [szew.essbase.otl :as ess.otl]))

(defn retest []
  (run-tests (find-tests "test")))

;; Outlines

(defn outlines!
 ([fname]
  (def otl fname)

  (def dimension-list (ess.otl/list-dimensions otl))

  (def dimensions (ess.otl/extract-dimensions otl))

  (def lut (ess.otl/member-lut otl))

  (def lut-pkg (ess.otl/member-lut-package otl))
  :done)
 ([]
  (outlines! "datasets/sample_basic.xml")))

;; BSO

(defn block-storage! []
  (when-not (bound? #'lut-pkg)
    (outlines!))

  (let [bso "datasets/sample_basic_level0.txt"
        tnc (fn [[coords value]]
              (and (= "Texas" (coords "Market"))
                   (= "COGS" (coords "Measures"))))]

    (def b-cells (ess.bso/comb-cells lut-pkg (partial filterv tnc) bso))
    :done))

;; TxL

(defn trails! []
  (let [p (comp ess.txl/printable
                (partial filterv (ess.txl/user-matches? #"johny")))
        alg "datasets/sample.alg"
        atx "datasets/sample.atx"]

    (def txl (io/in! (ess.txl/tx-log {:processor p}) {:alg alg :atx atx}))
    :done))

;; MaxL spool

(defn spools! []
  (def spools (io/in! (ess.logs/maxl-spool {:processor vec, :column-width 256})
                      "datasets/maxl-output.log"))
  :done)
```

## Known limitations

This library is built on things I encountered in the wild.

### No "Duplicate Member" support yet

I've never worked with outlines that had "duplicate members" enabled, therefore
handling of these was never implemented. Member LUT and data processing assumes
that all Member names are Dimension unique -- having shared members is fine,
having the same name mean different things across dimensions will be trouble.

If anyone is interested in having this -- hit me up with outline and data
samples, I'll take a crack at it.

### Illegal Reflection warning in XML processing

Unfortunately `data.xml` causes JDK9+ to have a fit about some reflections.
Clojure FAQ [Why do I get an illegal access warning?][FAQia] entry uses
the very call as example. Adding `--illegal-access=deny` is recommended until
this is fixed in upstream.

### Breaking Changes

Until version 1.0.0 is reached minor version upgrades may introduce
incompatible changes and deprecations. These will be clearly listed in
[changelog][changelog] and reflected in the [API documentation][latest].

* Breaking changes in `szew.essbase.txl` in 0.3.0.
* Multiple things broken and deprecated in 0.2.0

## License

Copyright © 2012-2020 Sławek Gwizdowski

**MIT License, text can be found in the LICENSE file.**

[latest]: http://spottr.bitbucket.io/szew-essbase/latest/
[changelog]: CHANGELOG.md
[FAQia]: https://clojure.org/guides/faq#illegal_access
