; Copyright (c) Sławek Gwizdowski
;
; Permission is hereby granted, free of charge, to any person obtaining
; a copy of this software and associated documentation files (the "Software"),
; to deal in the Software without restriction, including without limitation
; the rights to use, copy, modify, merge, publish, distribute, sublicense,
; and/or sell copies of the Software, and to permit persons to whom the
; Software is furnished to do so, subject to the following conditions:
;
; The above copyright notice and this permission notice shall be included
; in all copies or substantial portions of the Software.
;
; THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
; OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
; FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
; THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
; LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
; FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
; IN THE SOFTWARE.
;
(ns ^{:author "Sławek Gwizdowski"
      :doc "H2 database helper functions."}
 szew.h2
 (:require
   [clojure.java.jdbc :as jdbc]
   [clojure.string :as string]
   [clojure.java.io :as clj.io]
   [clojure.spec.alpha :as s])
 (:import
   [org.h2.jdbcx JdbcDataSource]
   [java.io File]))


;; ## Connection storage
;;
;; Just to make life easier we'll keep an atom with all the connection
;; specs and hopefully you'll be responsible about it. It's defonce,
;; so it survives reloading.


(defonce
  ^{:doc "Database connection storage."}
  connections (atom (hash-map)))


;; ## Default settings used by spec


(s/def ::method #{:default :nio :raw :tcp :mem :nio-mem-fs :nio-mem-lzf :raf})


(s/def ::user string?)


(s/def ::password string?)


(s/def ::hostname string?)


(s/def ::port (s/and int? pos?))


(s/def ::split? boolean?)


(s/def ::part (s/and int? pos?))


(s/def ::flags (s/map-of string? string?))


(s/def ::opts
  (s/keys :opt-un [::method ::user ::password ::hostname ::port ::split?
                   ::part ::flags]))


(def
  ^{:doc "Settings used as base for user spec."}
  defaults
  {:method   :default
   :user     "sa"
   :password ""
   :hostname "localhost"
   :port     9092
   :split?   false
   :part     31 ;; 2GiB, 2**31 bytes
   :flags    {"COMPRESS"      "TRUE"
              "EARLY_FILTER"  "TRUE"
              "DEFRAG_ALWAYS" "FALSE"
              ; disable BETA features, use old engine
              "MV_STORE"      "FALSE"
              "MVCC"          "FALSE"
              }})


(s/def ::classname (partial = "org.h2.Driver"))


(s/def ::subprotocol (partial = "h2"))


(s/def ::subname string?)


(s/def ::spec
  (s/keys :req-un [::classname ::subprotocol ::subname ::user ::password]))


(defn spec
  "Create H2 spec for given path (required argument).

  Store depends on method parameter of opts:

  * :raw sets :subname to given path verbose (best of luck!)
  * :tcp uses server, port and path for :subname
  * :mem sets database to in-memory storage, JVM runtime persistent
  * :nio-mem-fs is the new in-memory storage, JVM runtime persistent
  * :nio-mem-lzf for new in-memory compressing storage, JVM runtime persistent
  * :raf sets database to embedded, RandomAccessFile local storage
  * :default or anything else means local embedded (nio) file storage.

  Other opts are :user, :password, :server & :port. Self explanatory.

  For split filesystem use :split? flag and (2 ** :part) bytes for size.

  :flags should be a map of {String String}, like {COMPRESS TRUE}, it will
  overwrite flags set by default so YMMV.

  All in-memory stores are opened with DB_CLOSE_DELAY=-1, which means they will
  be available until JVM shuts down. You can close them with SHUTDOWN command.
  "
  ([path opts]
   {:pre  [(string? path) (s/valid? (s/nilable ::opts) opts)]
    :post [(s/valid? ::spec %)]}
   (let [merged   (merge defaults opts)
         {:keys [method user password hostname port flags split? part]} merged
         tail    (if (seq flags)
                   (->> flags
                        (map (partial apply (partial format "%s=%s")))
                        (string/join ";")
                        (str ";"))
                   "")]
     {:classname   "org.h2.Driver"
      :subprotocol "h2"
      :subname     (case method
                     ; ganbare!
                     :raw
                     path
                     ; remote H2 instance
                     :tcp
                     (format "tcp://%s:%d/%s%s" hostname port path tail)
                     ; in-memory alive until JVM exits
                     :mem
                     (format "mem:%s;DB_CLOSE_DELAY=-1%s" path tail)
                     ; nio in-memory alive until JVM exits
                     :nio-mem-fs
                     (format "nioMemFS:%s;DB_CLOSE_DELAY=-1%s" path tail)
                     ; compressing nio in-memory alive until JVM exits
                     :nio-mem-lzf
                     (format "nioMemLZF:%s;DB_CLOSE_DELAY=-1%s" path tail)
                     ; RandomAccessFile based database (non-NIO)
                     :raf
                     (if split?
                       (format "split:%d:%s%s" part path tail)
                       (format "%s%s" path tail))
                     ; standard in-file persistence using NIO
                     (if split?
                       (format "split:%d:nio:%s%s" part path tail)
                       (format "nio:%s%s" path tail)))
      :user        user
      :password    password}))
  ([path]
   {:pre  [(string? path)]
    :post [(s/valid? ::spec %)]}
   (spec path nil)))


(s/fdef spec
  :args (s/cat :path string? :spec (s/nilable ::spec))
  :ret  ::spec)


(defn spec->datasource
  "Eats spec, returns {:datasource org.h2.jdbcx.JdbcDataSource}.

  Requires :subname, :user and :password.
  "
  [a-spec]
  {:pre  [(s/valid? ::spec a-spec)]}
  (let [ds (doto (JdbcDataSource.)
             (.setURL (str "jdbc:h2:" (:subname a-spec)))
             (.setUser (:user a-spec))
             (.setPassword (:password a-spec)))]
    {:datasource ds}))


(defn make!
  "Create pool entry using given spec. Name defaults to :default.

  If spec-or-pool is nil, then this pool is released from storage.

  If spec-or-pool is a Delay, then it's stored directly and returned.
  "
  ([spec-or-pool a-name]
   (if (nil? spec-or-pool)
     (do (swap! connections dissoc a-name) nil)
     (let [conn (if (delay? spec-or-pool)
                  spec-or-pool
                  (delay spec-or-pool))]
       (swap! connections assoc a-name conn)
       conn)))
  ([spec-or-pool]
   (make! spec-or-pool :default)))


(defn conn!
  "Select named connection or :default. Throws ex-info if no such connection.
  "
  ([a-name]
   (if-let [a-pool (get @connections a-name)]
     @a-pool
     (throw (ex-info (format "Pool '%s' not found!" a-name)
                     {:name a-name
                      :available (into (hash-set) (keys @connections))}))))
  ([]
   (conn! :default)))


;; ## Misc helpers


(defn de-clob
  "CLOB to String conversion in a-map returned from the database.

  Warning: database connection must be open -- run in transaction.
  "
  ([a-map touch-keys]
   (->> (select-keys a-map touch-keys)
        (filter (comp (partial instance? java.sql.Clob) val))
        (map (fn [[key ^java.sql.Clob clob]]
               (with-open [r (clj.io/reader (.getCharacterStream clob))]
                 [key (slurp r)])))
        (into a-map)))
  ([a-map]
   (de-clob a-map (keys a-map))))


(defn columnlist?
  "Are we being asked for columnlist spec?

  This will return true if inside a function being used as stored procedure
  by H2 engine. It means that we are requested to return java.sql.ResultSet,
  like org.h2.tools.SimpleResultSet (for queries) or Java type mapped by H2
  for functions.
  "
  [^java.sql.Connection conn]
  (let [url (.getURL (.getMetaData conn))]
    (= "jdbc:columnlist:connection" url)))


;; ## H2 seeding functions.


(defn local!
  "Create a database spec for locally stored database.
  "
  ([path a-name]
   (Thread/sleep 100)
   (-> path
       (spec)
       (make! a-name)))
  ([path]
   (local! path :default)))


(defn in-memory!
  "Create a database spec for in-memory, JVM runtime persistent database.
  "
  ([path a-name]
   (-> path
       (spec {:method :mem})
       (make! a-name)))
  ([path]
   (in-memory! path :default)))


;; ## Insert and read modes.


(defn bulk-insert-mode!
  "Set H2 into bulk insert mode recommended in docs.
  "
  [conn]
  (jdbc/execute! conn
                 ["SET UNDO_LOG 0;
                   SET LOG 0;
                   SET LOCK_MODE 1;
                   SET WRITE_DELAY 0;
                   SET COMPRESS_LOB DEFLATE;"]))


(defn retrieval-mode!
  "Set H2 into settings closer to defaults (logging etc.).
  "
  [conn]
  (jdbc/execute! conn
                 ["SET UNDO_LOG 1;
                   SET LOG 2;
                   SET LOCK_MODE 3;
                   SET WRITE_DELAY 250;
                   SET COMPRESS_LOB DEFLATE;"]))


;; ## Maintenance tools


(defn dump!
  "Exports database into a SQL file (server side).
  "
  ([conn dump-file compression?]
   (io! "IO happens."
        (if compression?
          (jdbc/query conn
                      ["SCRIPT TO ? COMPRESSION GZIP;" dump-file]
                      {:result-set-fn vec})
          (jdbc/query conn
                      ["SCRIPT TO ?;" dump-file]
                      {:result-set-fn vec})))))


(defn pump!
  "Imports database from SQL file (server side).
  "
  ([conn dump-file compression?]
   (io! "IO happens."
        (if compression?
          (jdbc/execute! conn ["RUNSCRIPT FROM ? COMPRESSION GZIP;" dump-file])
          (jdbc/execute! conn ["RUNSCRIPT FROM ?;" dump-file])))))


(defn raze!
  "Drops all objects and, if no connections are open, deletes database files.

  Usually in embedded mode executing this removed the database files.
  "
  ([conn]
   (io! "IO happens."
        (jdbc/execute! conn ["DROP ALL OBJECTS DELETE FILES;"]))))

(defn shutdown!
  "Executes SHUTDOWN in connections.
  "
  ([conn]
   (io! "IO happens."
        (jdbc/execute! conn ["SHUTDOWN;"]))))

(defn copy!
  "Will dump! source, raze! target, pump! target.

  If dump-file not given uses temporary file and deletes it afterwards.
  "
  ([source target dump-file compression?]
   (io! "IO on top of IO."
        (dump! source dump-file compression?)
        (raze! target)
        (pump! target dump-file compression?)))
  ([source target compression?]
   (io! "IO on top of IO."
        (let [storage   (File/createTempFile "szew-copy-" ".bin")
              dump-file (.getCanonicalPath ^File storage)]
          (try
            (dump! source dump-file compression?)
            (raze! target)
            (pump! target dump-file compression?)
            (finally (.delete ^File storage)))))))
