# szew/h2

Clojure wrapper for H2 database.

[![szew/io](https://clojars.org/szew/h2/latest-version.svg)](https://clojars.org/szew/h2)

[API Codox][1]

## Why

I've been dogfooding my private Clo*j*ure toolbox (named `szew`) since 2012.

Splitting out and releasing non-proprietary parts.

## Usage

Conveniently `spec`ify H2 database connection map or datasource for
`org.clojure/java.jdbc`. Spec map:

```clojure
user=> (require '[szew.h2 :as h2])
nil
user=> (h2/spec "dbname")
{:classname "org.h2.Driver"
 :password ""
 :subname "nio:dbname;COMPRESS=TRUE;(...)"
 :subprotocol "h2"
 :user "sa"}
user=> (h2/spec "dbname" {:method :mem})
{:classname "org.h2.Driver"
 :password ""
 :subname "mem:dbname;DB_CLOSE_DELAY=-1;(...)"
 :subprotocol "h2"
 :user "sa"}
user=> (h2/spec "dbname" {:method :tcp})
{:classname "org.h2.Driver"
 :password ""
 :subname "tcp://localhost:9092/dbname;COMPRESS=TRUE;(...)"
 :subprotocol "h2"
 :user "sa"}

```

URL based datasource:

```clojure
user=> (h2/spec->datasource (h2/spec "dbname"))
{:datasource #object[org.h2.jdbcx.JdbcDataSource 0x9304380 "..."]}
```

Second argument, the `opts` map is:

```clojure
{:method   :default  ;; :raw, :tcp, :raf, :mem, :nio-mem-fs, :nio-mem-lzf
 :user     "sa"
 :password ""
 :hostname "localhost"
 :port     9092
 :split?   false
 :part     31
 :flags    {"OPTION" "VALUE"}}
```

The `:default` method is NIO.

Multiple connections can be stored in global `atom` called `connections`,
terms and (race) conditions apply:

```clojure
user=> (h2/make! (h2/spec "mem1" {:method :mem}) :mem1)
#<Delay@7fac2bf3 pending>
user=> (h2/make! (h2/spec "mem2" {:method :mem}) :mem2)
#<Delay@6c05d056 pending>
user=> (h2/conn! :mem1)
{:classname "org.h2.Driver"
 :password ""
 :subname "mem:mem1;DB_CLOSE_DELAY=-1;(...)"
 :subprotocol "h2"
 :user "sa"}
user=> (h2/conn! :mem2)
{:classname "org.h2.Driver"
 :password ""
 :subname "mem:mem2;DB_CLOSE_DELAY=-1;(...)"
 :subprotocol "h2"
 :user "sa"}
```

There's a shortcut for these, called `in-memory!`:

```clojure
user=> (h2/in-memory! "mem3" :mem3)
#<Delay@34dbb5c5 pending>
user=> (h2/conn! :mem3)
{:classname "org.h2.Driver"
 :password ""
 :subname "mem:mem3;DB_CLOSE_DELAY=-1;(...)"
 :subprotocol "h2"
 :user "sa"}
```

And if you only need one quick connection it's this short:

```clojure
user=> (h2/in-memory! "mem")
#<Delay@340dc7ed pending>
user=> (h2/conn!)
{:classname "org.h2.Driver"
 :password ""
 :subname "mem:mem;DB_CLOSE_DELAY=-1;(...)"
 :subprotocol "h2"
 :user "sa"}
user=> (h2/dispose!)
:disposed!
```


### What else?

Some helpers like `dump!`, `pump!`, `raze!`, `shutdown!` and `copy!`.

## Bonus: Extending H2 with Clojure

H2 can be extended with Java. We can ride on that with interop. To add
functions and procedure to H2 you need something like:

```clojure
(ns my-procedures
  (:gen-class
    :name "my_procedures"
    :main false
    :methods [
     ^{:static true}
     [long2str [java.lang.Long] java.lang.String]
     ^{:static true}
     [xnay [java.sql.Connection java.lang.Long] java.sql.ResultSet]]))


;; This will be a function
(defn -long2str [a-long]
  (str a-long))

;; This will be more like a stored procedure
(defn -xnay [^java.sql.Connection conn ^String q]
  ;;java.sql.ResultSet is expected here
  (.executeQuery (.createStatement conn) q))
```

Then compile and register in H2 via SQL:

```sql
CREATE ALIAS F FOR "my_procedures.long2str";
CREATE ALIAS P FOR "my_procedures.xnay";
```

And call with SQL:

```sql
CALL F(100);
-- => "100"
CALL P('SELECT * FROM SOMETABLE');
-- => results of that select
```

## License

Copyright © 2012 - 2019 Sławek Gwizdowski

MIT License, text can be found in the LICENSE file.

[1]: http://spottr.bitbucket.io/szew-h2/latest/

