(ns tailrecursion.boot.gitignore
  (:refer-clojure :exclude [replace])
  (:require
    [clojure.java.io  :refer [file]]
    [clojure.string   :refer [blank? join split replace]]
    )
  (:import
    [java.nio.file FileSystems]
    )
  )

(defprotocol IMatcher
  (-negated? [this] "Is this pattern negated?")
  (-matches? [this f] "Does file f match this pattern?"))

(defrecord Matcher [negated? matcher]
  IMatcher
  (-negated? [this] negated?)
  (-matches? [this f] (matcher f)))

(defn matches? [matchers f]
  (loop [match? nil, [matcher & more-matchers] matchers]
    (if-not matcher
      match?
      (let [m?  (-matches? matcher f)
            n?  (-negated? matcher)] 
        (recur (if (not m?) match? (not n?)) more-matchers)))))

(defn path-matcher [pattern & [negated?]]
  (let [m (.. FileSystems (getDefault) (getPathMatcher (str "glob:" pattern)))]
    (Matcher. negated? (fn [f] (.matches m (.toPath (.getCanonicalFile f)))))))

(defn parse-gitignore1 [pattern base]
  (let [base  (if (.endsWith base "/") base (str base "/"))
        strip #(replace % #"^/*" "")
        pat   (atom pattern)
        mat   (atom [])
        [negated? end-slash? has-slash? lead-slash? lead-asts? end-ast?] 
        (map #(fn [] (re-find % @pat)) [#"^!" #"/$" #"/" #"^/" #"^\*\*/" #"/\*$"])]
    (when (end-slash?) 
      (swap! pat #(subs % 0 (dec (count %))))
      (swap! mat conj #(.isDirectory %)))
    (if-not (has-slash?) 
      (swap! mat into (map path-matcher [(str base @pat) (str base "**/" @pat)]))
      (swap! mat conj (path-matcher (str base (strip @pat)))))
    (when (lead-asts?)
      (swap! mat conj (path-matcher (str base (strip (subs @pat 3))))))
    (when (end-ast?)
      (swap! mat conj (path-matcher (str base "/")))
      )
    (fn [f]
      (if (and end-slash? (not (.isDirectory f)))
        false
        true

        )
      )))

(defn parse-gitignore [f]
  (let [skip? #(or (blank? %) (re-find #"^\s*#" %))
        lines (->> f slurp (split #"\n") (remove skip?))
        ])
  )
