(ns tailrecursion.javastar
  (:require
   [alandipert.interpol8 :refer [interpolating]])
  (:import
   [javax.tools JavaCompiler SimpleJavaFileObject ToolProvider JavaFileObject$Kind]))

(defn source-object
  "Returns a JavaFileObject to store a class file's source."
  [class-name source]
  (proxy [SimpleJavaFileObject]
      [(java.net.URI/create (str "string:///"
                                 (.replace ^String class-name \. \/)
                                 (. JavaFileObject$Kind/SOURCE extension)))
       JavaFileObject$Kind/SOURCE]
    (getCharContent [_] source)))

(defn class-object
  "Returns a JavaFileObject to store a class file's bytecode."
  [class-name baos]
  (proxy [SimpleJavaFileObject]
      [(java.net.URI/create (str "string:///"
                                 (.replace ^String class-name \. \/)
                                 (. JavaFileObject$Kind/CLASS extension)))
       JavaFileObject$Kind/CLASS]
    (openOutputStream [] baos)))

(defn class-manager
  "An in-memory JavaFileManager for storing generated bytecode."
  [manager]
  (let [baos (java.io.ByteArrayOutputStream.)]
    (proxy [javax.tools.ForwardingJavaFileManager] [manager]
      (getClassLoader [location]
        (proxy [clojure.lang.DynamicClassLoader] []
          (findClass [name]
            (proxy-super defineClass name (.toByteArray baos) nil))))
      (getJavaFileForOutput [location class-name kind sibling]
        (class-object class-name baos)))))

(defn compile-java
  "Compiles the class-name implemented by source and loads it,
  returning the loaded Class."
  [class-name source]
  (let [compiler (ToolProvider/getSystemJavaCompiler)
        mgr (class-manager (.getStandardFileManager compiler nil nil nil))
        task (.getTask compiler nil mgr nil nil nil [(source-object class-name source)])]
    (if (.call task)
      (.loadClass (.getClassLoader mgr nil) class-name)
      (throw (RuntimeException. "Error compiling inline Java.")))))

(defn occurrences
  "Count of the occurrences of substring in s."
  ([s substring]
     (occurrences 0 s substring))
  ([n ^String s ^String substring]
     (let [i (.indexOf s substring)]
       (if (neg? i)
         n
         (recur (inc n) (.substring s (inc i)) substring)))))

(defn substitute
  "Replace pattern in s with substitutions."
  [s pattern substitutions]
  (reduce #(.replaceFirst ^String %1 pattern %2) s substitutions))

(defn generate-class
  "Generates, compiles, and loads a class with a single method, m,
  containing the code string.  Defines m's signature using
  return-type, arg-types, and ~{} occurrences in source as with js*.

  Returns the loaded class's name as a symbol."
  [imports return-type arg-types code]
  (let [class-name (str (gensym "tailrecursion_java_STAR_class"))
        n (occurrences code "~{}")
        arg-names (mapv str (repeatedly n gensym))
        arguments (->> (map #(str %1 " " %2) arg-types arg-names)
                       (interpose \,)
                       (apply str))
        method-body (substitute code "~\\{\\}" arg-names)
        prelude (apply str (map #(format "import %s;" (name %)) imports))
        class-body (interpolating
                    "#{prelude}
                     public class #{class-name} {
                       public static #{return-type} m (#{arguments}) {
                         #{method-body}
                       }
                     }")]
;    (println class-body)
   (compile-java class-name class-body)
   (symbol class-name)
    ))

(def prim-aliases
  "Type aliases for use with the return-type and arg-types arguments
   of java*."
  '{void     "void"
    boolean  "boolean"
    byte     "byte"
    char     "char"
    float    "float"
    int      "int"
    double   "double"
    long     "long"
    short    "short"
    booleans "boolean []"
    bytes    "byte []"
    chars    "char []"
    floats   "float []"
    ints     "int []"
    doubles  "double []"
    longs    "long []"
    shorts   "short []"})

(defn unalias
  "Attempts to resolve sym as a primitive alias or imported Java
  class."
  [sym]
  (or (get prim-aliases sym)
      (if-let [k (get (ns-imports *ns*) sym)] (.getName k))
      sym))

(defmacro java*
  "Similar to ClojureScript's js*.  Compiles a Java code string into a
  Java method and invokes the method with args.

  java* has more arguments than js*.  imports is a vector of
  zero or more fully qualified class names.  return-type and arg-types
  may be either Java classes or symbol aliases for primitive types and
  arrays.  See prim-aliases for available aliases.

  Example:

  (def java-add #(java* [] long [long long] \"return ~{} + ~{};\" %1 %2))
  (java-add 1 2) ;=> 3"
  [imports return-type arg-types code & args]
  (let [g (generate-class imports (unalias return-type) (map unalias arg-types) code)]
    `(. ~g ~'m ~@args)))
