;   Copyright (c) Alan Dipert and Micha Niskin. All rights reserved.
;   The use and distribution terms for this software are covered by the
;   Eclipse Public License 1.0 (http://opensource.org/licenses/eclipse-1.0.php)
;   which can be found in the file epl-v10.html at the root of this distribution.
;   By using this software in any fashion, you are agreeing to be bound by
;   the terms of this license.
;   You must not remove this notice, or any other, from this software.

(ns tailrecursion.javelin-clj
  (:require
   [clojure.data.priority-map :refer [priority-map]]
   [clojure.walk    :refer [macroexpand-all prewalk]]
   [clojure.java.io :as io]
   [clojure.string  :as s]))

;;; core

(defn bf-seq [branch? children root]
  (let [walk (fn walk [queue]
               (when-let [node (peek queue)]
                 (lazy-seq
                  (cons node (walk (into (pop queue)
                                         (when (branch? node)
                                           (children node))))))))]
    (walk (conj clojure.lang.PersistentQueue/EMPTY root))))

(declare input)

(defrecord CellMeta [rank prev sources sinks done thunk])

(defn getmeta [obj k]
  (-> obj meta ::cell deref k))

(defn cell? [obj]
  (boolean (-> obj meta ::cell)))

(defn alter! [cell f & args]
  (dosync (alter (-> cell meta ::cell) #(apply f % args)))
  cell)

(defn done? [cell]
  (boolean (getmeta cell :done)))

(let [rank (atom 0)]
  (defn next-rank [] (swap! rank inc)))

(defn deref* [x] (if (cell? x) @x x))

(defn self? [x] (and (cell? x) (map? @x) (contains? @x ::self)))

(defn sub-self [this xs]
  (map #(if (self? %) (if (= ::none @this) (input (::self @%)) this) %) xs))

(defn sinks-seq [c]
  (tree-seq cell? #(seq (getmeta % :sinks)) c))

;;; TODO fix dosync scope
(defn propagate! [cell]
  (loop [queue (priority-map cell (getmeta cell :rank))]
    (when (seq queue)
      (let [next      (key (peek queue))
            value     ((getmeta next :thunk))
            continue? (not= value (getmeta next :prev))
            reducer  #(assoc %1 %2 (getmeta %2 :rank))
            siblings  (pop queue)
            children  (getmeta next :sinks)]
        (if continue? (alter! next assoc :prev value))
        (recur (if continue? (reduce reducer siblings children) siblings))))))

;;; TODO fix dosync scope
(defn set-formula! [this & [f sources]]
  (doseq [source (filter cell? (getmeta this :sources))]
    (alter! source update-in [:sinks] disj this))
  (alter! this assoc :sources (if f (conj (vec sources) f) (vec sources)))
  (doseq [source (filter cell? (getmeta this :sources))]
    (alter! source update-in [:sinks] conj this)
    (if (> (getmeta source :rank) (getmeta this :rank))
      (doseq [dep (bf-seq identity :sinks source)]
        (alter! dep assoc :rank (next-rank)))))
  (let [compute #(apply (deref* (peek %)) (map deref* (sub-self this (pop %))))
        thunk   #(dosync (ref-set this (compute (getmeta this :sources))))]
    (if f (remove-watch this ::propagate)
        (add-watch this ::propagate (fn [_ cell _ _] (propagate! cell))))
    (alter! this assoc :thunk (if f thunk #(deref this)))
    (doto this propagate!)))

(def self   #(input {::self %}))
(def input* #(if (cell? %) % (input %)))

(defn input [value]
  (set-formula!
   (ref value
        :meta {::cell (ref (CellMeta. (next-rank) value [] #{} false nil))})))

(defn lift [f]
  (fn [& sources]
    (set-formula! (input ::none) f sources)))

;;; cell macro

(defn to-list [coll]
  (into () (reverse coll)))

(defn unquote?
  [form]
  (and (seq? form)
       (= 'clojure.core/unquote (first form))))

(defn quoted?
  [form]
  (and (seq? form)
       (= 'quote (first form))))

(defn func?
  [form]
  (and (seq? form)
       (= 'fn* (first form))))

(defn let*?
  [form]
  (and (seq? form)
       (= 'let* (first form))))

(defn dot?
  [form]
  (and (seq? form)
       (= '. (first form))))

(defn listy?
  [form]
  (or (list? form)
      (= clojure.lang.LazySeq (type form))
      (= clojure.lang.Cons (type form))))

(declare do-lift)

(defn do-map
  [form]
  (do-lift (to-list `(hash-map ~@(mapcat identity form)))))

(defn do-vector
  [form]
  (do-lift (to-list `(vector ~@form))))

(defn do-set
  [form]
  (do-lift (to-list `(set ~@form))))

(defn do-let*
  [[_ bindings & body]]
  `(let* ~(mapv #(%1 %2) (cycle [identity do-lift]) bindings)
         ~@(map do-lift body)))

(defn do-self
  [form]
  `(input ~{::self `(deref* ~(do-lift (second form)))}))

(defn do-dot
  [[_ obj meth & args]]
  (let [bindings (map (fn [_] (gensym)) args)]
    (do-lift `((fn* [obj# ~@bindings] (~'. obj# ~meth ~@bindings)) ~obj ~@args))))

;;; TODO: include all Clojure specials
(def specials #{'if 'def 'do 'loop* 'letfn* 'throw 'try 'recur 'new 'set!
                'ns 'deftype* 'defrecord* '& 'monitor-enter 'monitor-exit
                'case*})

(defn special?
  [op]
  (boolean (get specials op)))

(defn do-special
  [op]
  (condp = op
    'do `(fn [& forms#] (last forms#))
    'if `(fn
           ([pred# then#] (if pred# then#))
           ([pred# then# else#] (if pred# then# else#)))
    ;; TODO: new
    (throw (IllegalArgumentException. (str op " is not supported in cell formulas.")))))

(defn do-lift
  [form]
  (cond
   (map? form)         (do-map form)
   (vector? form)      (do-vector form)
   (set? form)         (do-set form)
   (not (listy? form)) form
   (unquote? form)     (do-self form)
   (quoted? form)      (second form)
   (func? form)        `(input form)
   (let*? form)        (do-let* form)
   (dot? form)         (do-dot form)
   :else               (let [[op & args] form]
                         (if (= op 'apply)
                           `(apply (lift ~(do-lift (first args))) ~@(map do-lift (rest args)))
                           `((lift ~(if (special? op)
                                      (do-special op)
                                      (do-lift op)))
                             ~@(map do-lift args))))))

(defmacro cell
  [form]
  (let [form    (macroexpand-all form)
        lifted  (do-lift form)
        q?      (or (quoted? form) (not (listy? lifted)))
        expr    (if q? (list input lifted) lifted)]
    expr))
