(ns tandem.app
  (:use net.cgrand.moustache)
  (:require [tandem.server :as server]))

(defonce ^:dynamic *servers* (ref {}))

(defn start-server
  "Start a server and record that it's running (so we can stop it layer, duh).
  The server is associated with the name of the application, defined by the
  defapp macro.

  Don't call this function directly; it is used by the defapp macro."
  [servers app port]
  (let [existing (servers port)]
    (if existing
      (throw (Exception. (str "A server is already running on port " port)))
      (conj servers {port (server/bootstrap app port)}))))

; (wrap-reload-modified app ["src" "resources/views"])

(defn stop-server
  "Stop the server running on the given port.  If a server is not running on
  that port, the request is ignored.

  Don't call this function directly; it is used by the defapp macro."
  [servers port]
  (let [existing (servers port)]
    (when existing
      (.disconnect existing)
      (.close existing))
    (dissoc servers port)))

(defmacro defapp
  "Create a web application in a namespace.  For example:

    (ns myapp.server
      (:use tandem.app))

    (defweb myweb
      [\"\"] {:get home/page}
      [\"user\" login] {:get user/info})

  Uses the Moustache (https://github.com/cgrand/moustache) library for routing.

  Creates a Ring application, wrapped with params, multipart, sessions and
  cookies.  Static resources are expected in resources/static.

  The Ring handlers you point Moustache at should return the standard Ring
  response, i.e. {:keys [status headers body]}.  An extension to the standard
  is that if a map ({}) is returned for the body, it is converted to a JSON
  string, the content type is set to 'application/json', and the JSON is
  returned.

  *** TODO:  Better description of routing!!! ***

  The web application will also automatically include the functionality
  typically specified in these Ring middleware libraries (in other words, you
  don't need to include them yourself):

    ring.middleware.resource
    ring.middleware.file-info
    ring.middleware.content-type
    ring.middleware.params
    ring.middleware.multipart-params
    ring.middleware.cookies
    ring.middleware.session
    ring.middleware.stacktrace

  Currently this functionality cannot be disabled, but it is the least of
  worries in terms of performance, as Netty deals with most of it automatically
  to begin with.

  Session stores can be configured in the config.json file.

  Logging is also included, similar to Rails logging.

  We do NOT support keywordized params.  Despite the improvements of Clojure
  1.3 with regards to keywords overloading the PermGen space, it still seems
  excessive to tax Keyword class creation from random user-generated terms.

  Once created, you can then call start and stop on your application
  to fire up a Netty server instance:

    (require 'myapp.server)
    (myapp.server/start)
    (myapp.server/stop)

  Each function takes an option port argument.  For example:

    (myapp.server/start 3000)
    (myapp.server/stop 3000)

  If you start an app on a specific port, make sure you restart and stop the
  app using the same port.  By default, the port is 8080."
  [name & routes]
  `(let [start# (fn [& [port#]]
                  (let [app# (app ~@routes
                                  [&] {:any server/page-not-found})]
                    (dosync (alter *servers* start-server app# (or port# 8080)))))
         stop# (fn [& [port#]] (dosync (alter *servers* stop-server (or port# 8080))))]
     (intern *ns* (symbol "start") start#)
     (intern *ns* (symbol "stop") stop#)))
