(ns tandem.http.headers
  "Functions around headers."
  (:refer-clojure :exclude [from get])
  (:require [clojure.string :as str]
            [tandem.http.header :as header]
            [tandem.http.charset :as charset]))

(def ^:dynamic *headers* nil)

(defn get
  "Use this in the with-headers macro forms to get header values."
  ([name default] (*headers* name default))
  ([name] (get name nil)))
  
(defn character-encoding
  "Determine the character encoding.  Defaults to UTF-8."
  ([headers] (headers header/content-encoding charset/utf-8))
  ([] (character-encoding *headers*)))

(defn content-type
  "Parse the Content-Type header and break out the content type, e.g.
  application/json"
  ([headers]
     (if-let [value (headers header/content-type)]
       (-> (str/split value #";")
           first
           str/trim
           str/lower-case)))
  ([] (content-type *headers*)))

(defn accept
  "Parse the Accept header and break out the accepted types, e.g. text/html,
  text/plain.  This greatly simplifies the accepted headers, and ignores Accept
  params like q and mxb."
  ([headers]
     (if-let [value (headers header/accept)]
       (map #(first (re-find #"([^;\s]*)" %)) (str/split value #"\s*,\s*"))))
  ([] (accept *headers*)))

(defn from
  "Parse the headers out of the given Netty channel message.  Returns a Ring-
  compatible map of lower-case header names to values."
  [message]
  (into {} (map (fn [entry]
                  [(-> entry .getKey .toLowerCase)
                   (.getValue entry)])
                (.getHeaders message))))

(defmacro with-headers
  "When accessing a number of headers, it can be convenient to wrap them in a
  with-headers macro, instead of always passing the headers around to each
  function."
  [headers & forms]
  `(binding [*headers* headers]
     ~@forms))

(defmacro with-headers-from
  "Like with-headers, but accepts a Netty channel message and loads the headers
  directly."
  [message & forms]
  `(binding [*headers* (from ~message)]
     ~@forms))
