(ns clj-google-maps.geocode
  (:require [clojure.string :as str]
            [clojure.spec.alpha :as spec]
            [clj-http.client :as http]
            [clj-google-maps.config :as config]))

;;
;; specs
;;

(spec/def ::address   (spec/and string? #(not (str/blank? %))))
(spec/def ::latitude  (spec/and number? #(<= -90 % 90)))
(spec/def ::longitude (spec/and number? #(<= -180 % 180)))
(spec/def ::location  (spec/tuple ::latitude ::longitude))
(spec/def ::place-id  (spec/and string? #(not (str/blank? %))))


;;
;; default settings
;;

(def ^:const ^:private url
  (str config/api-url "/" config/api-format))

(def ^:const ^:private query-params
  {:key config/api-key})


;;
;; geocoding api
;;

(defmulti fetch-by
  (fn [type & params] type))


(defmethod fetch-by :address [_ address]
  (if (spec/valid? ::address address)
    (try
      (let [params   {:query-params (merge query-params
                                           {:address  address
                                            :region   config/api-region
                                            :language config/api-language})}
            response (http/get url params)]
        (:body response))
      (catch Exception e
        (:body (ex-data e))))
    (let [message (format "Invalid address value. Explain data: %s" (spec/explain-data ::address address))]
      (throw (IllegalArgumentException. message)))))


(defmethod fetch-by :location [_ & location]
  (let [location' (vec location)]
    (if (spec/valid? ::location location')
      (try
        (let [latlng   (str/join "," location')
              params   {:query-params (merge query-params
                                             {:latlng   latlng
                                              :region   config/api-region
                                              :language config/api-language})}
              response (http/get url params)]
          (:body response))
        (catch Exception e
          (:body (ex-data e))))
      (let [message (format "Invalid location value. Explain data: %s" (spec/explain-data ::location location'))]
        (throw (IllegalArgumentException. message))))))


(defmethod fetch-by :place-id [_ place-id]
  (if (spec/valid? ::place-id place-id)
    (try
      (let [params   {:query-params (merge query-params
                                           {:place_id place-id
                                            :language config/api-language})}
            response (http/get url params)]
        (:body response))
      (catch Exception e
        (:body (ex-data e))))
    (let [message (format "Invalid place-id value. Explain data: %s" (spec/explain-data ::place-id place-id))]
      (throw (IllegalArgumentException. message)))))
