(ns clj-kafka.serialization.core
  (:require [clojure.string :as str]
            [clj-kafka.serialization.serializers :as serializers]
            [clj-kafka.serialization.deserializers :as deserializers]))


(def ^:private ^:const serialization-types
  #{:string :keyword :edn :json :transit})

(def ^:private ^:const transit-format-types
  #{:json :json-verbose :msgpack})

(def ^:private ^:const bad-serialization-type-message
  (str "Serialization type must be " (str/join ", " serialization-types)))

(def ^:private ^:const bad-transit-format-type-message
  (str "Transit format must be " (str/join ", " transit-format-types)))


(defn serializer [type & args]
  (if (contains? serialization-types type)
    (case type
      :string  (serializers/->StringSerializer)
      :keyword (serializers/->KeywordSerializer)
      :edn     (serializers/->EdnSerializer)
      :json    (let [[options]      args
                     mapper-options (select-keys options [:pretty :escape-non-ascii :date-format :encode-key-fn :encoders])]
                 (serializers/->JsonSerializer mapper-options))
      :transit (let [[format writer-options] args]
                 (when-not (contains? transit-format-types format)
                   (throw (IllegalArgumentException. bad-transit-format-type-message)))
                 (serializers/->TransitSerializer format writer-options)))
    (throw (IllegalArgumentException. bad-serialization-type-message))))


(defn deserializer [type & args]
  (if (contains? serialization-types type)
    (case type
      :string  (deserializers/->StringDeserializer)
      :keyword (deserializers/->KeywordDeserializer)
      :edn     (deserializers/->EdnDeserializer)
      :json    (let [[options]      args
                     mapper-options (select-keys options [:decode-key-fn])]
                 (deserializers/->JsonDeserializer mapper-options))
      :transit (let [[format reader-options] args]
                 (when-not (contains? transit-format-types format)
                   (throw (IllegalArgumentException. bad-transit-format-type-message)))
                 (deserializers/->TransitDeserializer format reader-options)))
    (throw (IllegalArgumentException. bad-serialization-type-message))))
