(ns clj-kafka.serialization.deserializers
  (:require [clojure.edn :as edn]
            [clojure.string :as str]
            [cognitect.transit :as transit]
            [jsonista.core :as jsonista])
  (:import [java.io ByteArrayInputStream]
           [org.apache.kafka.common.serialization Deserializer]))


(deftype StringDeserializer []
  Deserializer
  (configure [this configs is-key?])
  (deserialize [this topic data]
    (when (bytes? data)
      (String. data "UTF-8")))
  (close [this]))


(deftype KeywordDeserializer []
  Deserializer
  (configure [this configs is-key?])
  (deserialize [this topic data]
    (when (bytes? data)
      (keyword (String. data "UTF-8"))))
  (close [this]))


(deftype EdnDeserializer []
  Deserializer
  (configure [this configs is-key?])
  (deserialize [this topic data]
    (when (bytes? data)
      (edn/read-string (String. data "UTF-8"))))
  (close [this]))


(deftype JsonDeserializer [mapper-options]
  Deserializer
  (configure [this configs is-key?])
  (deserialize [this topic data]
    (when (bytes? data)
      (-> (String. data "UTF-8")
         (jsonista/read-value (jsonista/object-mapper mapper-options)))))
  (close [this]))


(deftype TransitDeserializer [format reader-options]
  Deserializer
  (configure [this configs is-key?])
  (deserialize [this topic data]
    (when (bytes? data)
      (-> (ByteArrayInputStream. data)
         (transit/reader format reader-options)
         (transit/read))))
  (close [this]))
