(ns clj-weather-underground.core
  (:require [clojure.string :as str]
            [clojure.spec.alpha :as spec]
            [clj-http.client :as http]
            [clj-weather-underground.config :as config]))

;;
;; default settings
;;

(def ^:const ^:private url
  (str config/api-url "/" config/api-key))

(def ^:const features
  #{:alerts :almanac :astronomy :conditions :current-hurricane :forecast
   :forecast-10day :geolookup :history :hourly :hourly-10day :planner
   :raw-tide :satellite :tide :webcams :yesterday})



;;
;; specs
;;


(spec/def ::feature    features)
(spec/def ::identifier (spec/and string? #(not (str/blank? %))))
(spec/def ::latitude   (spec/and number? #(<= -90 % 90)))
(spec/def ::longitude  (spec/and number? #(<= -180 % 180)))
(spec/def ::location   (spec/tuple ::latitude ::longitude))
(spec/def ::date       (spec/and string? #(re-matches #"^\d{8}$" %))) ;; TODO: use new java 8 datetime api for dates
(spec/def ::period     (spec/and string? #(re-matches #"^\d{4}$" %))) ;; TODO: check date period - 30 days max



;; TODO: add requests counter for handle current plan limits and scheduler for everyday reseting



;;
;; validators
;;

(defn- validate-feature! [feature]
  (when-not (spec/valid? ::feature feature)
    (let [message (format "Invalid feature value. Explain data: %s" (spec/explain-data ::feature feature))]
      (throw (IllegalArgumentException. message)))))


(defn- validate-identifier! [identifier]
  (when-not (spec/valid? ::identifier identifier)
    (let [message (format "Invalid identifier value. Explain data: %s" (spec/explain-data ::identifier identifier))]
      (throw (IllegalArgumentException. message)))))


(defn- validate-location! [location]
  (when-not (spec/valid? ::location location)
    (let [message (format "Invalid location value. Explain data: %s" (spec/explain-data ::location location))]
      (throw (IllegalArgumentException. message)))))


(defn- validate-date! [date]
  (when-not (spec/valid? ::date date)
    (let [message (format "Invalid date value. Explain data: %s" (spec/explain-data ::date date))]
      (throw (IllegalArgumentException. message)))))


(defn- validate-period! [period]
  (when-not (spec/valid? ::period period)
    (let [message (format "Invalid period value. Explain data: %s" (spec/explain-data ::period period))]
      (throw (IllegalArgumentException. message)))))



;;
;; helpers
;;

(defn- get-feature-url [feature identifier]
  (str url "/" feature "/q/" identifier "." config/api-format))



;;
;; weather underground api
;;
;; TODO: add response parsers

(defmulti fetch

  "Weather Underground API features implementations (https://www.wunderground.com/weather/api/d/docs?d=data/index)"

  (fn [feature & params]
    (validate-feature! feature)
    feature))



(defmethod fetch :alerts
  [_ identifier]

  "Returns the short name description, expiration time and a long text description of a severe alert, if one has been issued
for the searched upon location. These alerts are only active in the United States and Europe. View our United States Severe
Weather map or our European Severe Weather map to see where there are active alerts at this moment. Read the National Weather
Service description of VTEC codes used in severe weather alerts in the United States, including \"phenomena\" and \"significane\",
here http://www.weather.gov/os/vtec/pdfs/VTEC_explanation6.pdf.
European alerts are required to show the \"attribution\" to Meteoalarm.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/alerts
Request format: http://api.wunderground.com/api/YOUR_API_KEY/alerts/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "alerts" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :almanac
  [_ identifier]

  "Records within the United States come from the National Weather Service, with approximately 120 reporting stations giving
records. Records for the rest of the United States, and locations outside of the United States, come from the data we have stored.
These are compiled records and only go as far back as we have data for. The average high and low temperature going back as far as
Weather Underground has data for OR from National Weather Service going back 30 years.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/almanac
Request format: http://api.wunderground.com/api/YOUR_API_KEY/almanac/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "almanac" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :astronomy
  [_ identifier]

  "Returns the moon phase, sunrise and sunset times.

Documentation: https://www.wunderground.com/weather/api/d/docs?d=data/astronomy
Request format: http://api.wunderground.com/api/YOUR_API_KEY/astronomy/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "astronomy" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :conditions
  [_ identifier]

  "Returns the current temperature, weather condition, humidity, wind, 'feels like' temperature, barometric pressure, and visibility.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/conditions
Request format: http://api.wunderground.com/api/YOUR_API_KEY/conditions/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "conditions" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :current-hurricane
  [_ _]

  "Returns information about current hurricanes and tropical storms. This feature can be used with other weather API features.
However, location query options do not apply to the results for currenthurricane.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/currenthurricane
Request format: http://api.wunderground.com/api/YOUR_API_KEY/currenthurricane/view.json"

  (try
    (let [url'     (get-feature-url "currenthurricane" "view")
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :forecast
  [_ identifier]

  "Returns a summary of the weather for the next 3 days. This includes high and low temperatures, a string text forecast and the conditions.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/forecast
Request format: http://api.wunderground.com/api/YOUR_API_KEY/forecast/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "forecast" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :forecast-10day
  [_ identifier]

  "Returns a summary of the weather for the next 10 days. This includes high and low temperatures, a string text forecast and the conditions.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/forecast10day
Request format: http://api.wunderground.com/api/YOUR_API_KEY/forecast10day/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "forecast10day" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :geolookup
  [_ & location]

  "Returns the city name, zip code / postal code, latitude-longitude coordinates and nearby personal weather stations.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/geolookup
Request format: http://api.wunderground.com/api/YOUR_API_KEY/geolookup/q/LATITUDE,LONGITUDE.json"

  (let [location' (vec location)]
    (validate-location! location')
    (try
      (let [identifier (str/join "," location')
            url'       (get-feature-url "geolookup" identifier)
            response   (http/get url')]
        (:body response))
      (catch Exception e
        (:body (ex-data e))))))



(defmethod fetch :history
  [_ & [identifier date]]

  "Returns a summary of the observed weather for the specified date.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/history
Request format: http://api.wunderground.com/api/YOUR_API_KEY/history_YYYYMMDD/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (validate-date! date)
  (try
    (let [feature' (str "history_" date)
          url'     (get-feature-url feature' identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :hourly
  [_ identifier]

  "Returns a hourly of the weather for the next 3 days.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/hourly
Request format: http://api.wunderground.com/api/YOUR_API_KEY/hourly/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "hourly" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :hourly-10day
  [_ identifier]

  "Returns a hourly of the weather for the next 10 days.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/hourly10day
Request format: http://api.wunderground.com/api/YOUR_API_KEY/hourly10day/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "hourly10day" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :planner
  [_ & [identifier start end]]

  "Returns a weather summary based on historical information between the specified dates (30 days max).

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/planner
Request format: http://api.wunderground.com/api/YOUR_API_KEY/planner_MMDDMMDD/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (validate-period! start)
  (validate-period! end)
  (try
    (let [feature' (str "planner_" start end)
          url'     (get-feature-url feature' identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :raw-tide
  [_ identifier]

  "Raw Tidal information for graphs.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/rawtide
Request format: http://api.wunderground.com/api/YOUR_API_KEY/rawtide/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "rawtide" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :satellite
  [_ identifier]

  "Returns image url.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/satellite
Request format: http://api.wunderground.com/api/YOUR_API_KEY/satellite/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "satellite" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :tide
  [_ identifier]

  "Tidal information.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/tide
Request format: http://api.wunderground.com/api/YOUR_API_KEY/tide/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "tide" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :webcams
  [_ identifier]

  "Returns locations of nearby Personal Weather Stations and URLs for images from their web cams.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/webcams
Request format: http://api.wunderground.com/api/YOUR_API_KEY/webcams/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "webcams" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))



(defmethod fetch :yesterday
  [_ identifier]

  "Returns a summary of the observed weather history for yesterday.

Documentation:  https://www.wunderground.com/weather/api/d/docs?d=data/yesterday
Request format: http://api.wunderground.com/api/YOUR_API_KEY/yesterday/q/STATION_IDENTIFIER.json"

  (validate-identifier! identifier)
  (try
    (let [url'     (get-feature-url "yesterday" identifier)
          response (http/get url')]
      (:body response))
    (catch Exception e
      (:body (ex-data e)))))
