(ns tech.logging.core
  "A logging namespace for wrapping our logging impl.
  Uses Timbre to send your logs to a logstash or riemann.
  Configuration is derived from the config namespace."
  (:require [tech.config.core :as config]
            [clojure.string :as str]
            [io.aviso.exception :as aviso-ex]
            [taoensso.timbre :as timbre]
            [taoensso.timbre.appenders.core :refer [println-appender]]
            [tech.logging.riemann :as r]
            [tech.logging.logstash :as l]
            [tech.logging.pretty :as pretty]
            ;[think.trace :as trace]
            [clojure.string :as strs]
            [cheshire.core :refer [generate-string]]))

;; these first four lines add trace-ids to the console output of all
;; logging messages
(defn stacktrace [err & [{:keys [stacktrace-fonts] :as opts}]]
  (let [stacktrace-fonts (if (and (nil? stacktrace-fonts)
                                  (contains? opts :stacktrace-fonts))
                           {} stacktrace-fonts)]
    (if-let [fonts stacktrace-fonts]
      (binding [aviso-ex/*fonts* fonts] (aviso-ex/format-exception err))
      (aviso-ex/format-exception err))))

(defn default-std-output-fn
  "Default (fn [data]) -> string output fn.
  You can modify default options with `(partial default-output-fn
  <opts-map>)`."
  ([data] (default-std-output-fn nil data))
  ([{:keys [no-stacktrace? stacktrace-fonts] :as opts} data]
   (let [{:keys [level ?err_ vargs_ msg_ ?ns-str hostname_
                 timestamp_ ?line]} data]
     (str
       @timestamp_ " "
       @hostname_ " "
       (str/upper-case (name level)) " "
       ;trace/*trace-id* " "
       "[" (or ?ns-str "?") ":" (or ?line "?") "] - "
       (force msg_)
       (when-not no-stacktrace?
         (when-let [err (force ?err_)]
           (str "\n" (stacktrace err opts))))))))

(timbre/merge-config! {:output-fn (partial default-std-output-fn {})})

(defn add-exception-info
  [ex]
  (if (instance? Throwable ex)
    {:ex-message (.getMessage ^Throwable ex)
     :trace      (mapv str (.getStackTrace ^Throwable ex))}))

(defn add-defaults [a-map]
  (merge
    a-map

    ;(if-let [t trace/*trace-id*] {:trace-id t})

    ;; If :service is not specified but is in the config then add it
    (if-let [s (and (not (contains? a-map :service)) (config/unchecked-get-config :service))] {:service (name s)})

    ;; If :env is in the config then add it
    (if-let [env (config/unchecked-get-config :env)] {:env env})

    ;; If an :exception (or an :ex) is added take out the message and trace
    (if-let [e (:ex a-map)] (add-exception-info e)) (if-let [e (:exception a-map)] (add-exception-info e))))

(defn parse-log
  "Defines the formatting of the structured log messages. See the test for examples."
  [args & {:keys [translate-exception?] :or {translate-exception? true}}]
  (try
    (let [uno (first args)]
      (cond
        ;; The first arg can be an exception
        (instance? Throwable uno) (merge {:event "exception"}
                                         (if translate-exception?
                                           (add-exception-info uno)
                                           {:exception uno}))
        ;; Or a keyword, in which case this becomes the event
        (keyword? uno) (let [others (rest args)]
                         ;; For all args after the keyword
                         (cond
                           ;; Allow an even number of keyword value pairs
                           (and (even? (count others)) (every? keyword? (take-nth 2 others)))
                           (merge {:event (name uno)} (add-defaults (apply hash-map others)))
                           ;; Or a single map
                           (and (= 1 (count others)) (map? (first others)))
                           (merge {:event (name uno)} (add-defaults (first others)))
                           ;; Otherwise its a string
                           :else (merge {:event (name uno) :message (apply str others)} (add-defaults {}))))
        ;; Anything else gets str'ed
        :else (merge {:event :system :message (apply str args)} (add-defaults {}))))
    ;; And if everything goes horribly wrong
    (catch Throwable t (merge {:event :log-format-exception :args args} (add-exception-info t)))))


(defn default-output-fn
  [{:keys [level hostname_ ?ns-str vargs_ ?err_] :as all}]
  (merge
    {:level                 (name level)
     :ns                    ?ns-str
     :host                  (strs/upper-case @hostname_)
     (keyword "@timestamp") (.getTime (java.util.Date.))}
    (parse-log @vargs_)
    (if (and @?err_ (instance? Throwable @?err_))
      (merge {:event "exception"} (add-exception-info @?err_)))))

(defn pretty-output-fn
  [{:keys [level hostname_ ?ns-str vargs_ ?err_ ?line] :as all}]
  (merge
    {:level                 (name level)
     :ns                    (format "%s:%s" ?ns-str ?line)
     :host                  (strs/upper-case @hostname_)
     (keyword "@timestamp") (.getTime (java.util.Date.))}
    (parse-log @vargs_ :translate-exception? false)
    (if (and @?err_ (instance? Throwable @?err_))
      (merge {:event "exception" :exception @?err_}))))

(defn log-to-riemann []
  (println (format "Setting up riemann logging to: %s:%s"
                   (config/get-config :log-host)
                   (config/get-config :log-port true)))
  (timbre/merge-config!
    {:level  (config/get-config :log-level)
     :appenders  {:riemann (r/riemann-appender
                             {:host      (config/get-config :log-host)
                              :port      (config/get-config :log-port true)
                              :output-fn #'default-output-fn})}
     :middleware []}))

(defn log-to-pretty []
  (timbre/merge-config!
    {:level  (config/get-config :log-level)
     :appenders  {:println {:enabled? false}
                  :pretty  {:enabled?   true
                            :async?     false
                            :min-level  nil
                            :rate-limit nil
                            :output-fn  :inherit
                            :fn         (pretty/log-fn pretty-output-fn)}}
     :middleware []}))

(cheshire.generate/add-encoder
  java.io.File
  (fn [^java.io.File f ^com.fasterxml.jackson.core.JsonGenerator jg]
    (.writeString jg (.toString f))))

(cheshire.generate/add-encoder
  Throwable
  (fn [^Throwable f ^com.fasterxml.jackson.core.JsonGenerator jg]
    (.writeString jg (.getMessage f))))

(cheshire.generate/add-encoder
  java.lang.Object
  (fn [f ^com.fasterxml.jackson.core.JsonGenerator jg]
    (.writeString jg (str "Java Object [" (type f) "][" (str f) "]"))))

(defn log-to-logstash []
  (println (format "Setting up logstash logging to: %s:%s"
                   (config/get-config :log-host)
                   (config/get-config :log-port true)))
  (timbre/merge-config!
    {:level  (config/get-config :log-level)
     :appenders  {:logstash (l/logstash-appender
                              {:host      (config/get-config :log-host)
                               :port      (config/get-config :log-port true)
                               :output-fn #'default-output-fn})}
     :middleware []}))


(defn log-to-println []
  (println (format "Setting up logstash logging to: %s:%s"
                   (config/get-config :log-host)
                   (config/get-config :log-port true)))
  (timbre/merge-config!
    {:level  (config/get-config :log-level)
     :appenders  {:logstash (l/logstash-appender
                              {:host      (config/get-config :log-host)
                               :port      (config/get-config :log-port true)
                               :output-fn #'default-output-fn})}
     :middleware []}))

(defn add-appenders!
  [& {:keys [verbose]
      :or {verbose false}}]
  (try
    (case (config/get-config :log-system)
      "pretty" (log-to-pretty)
      "riemann" (log-to-riemann)
      "logstash" (log-to-logstash)
      "println" (timbre/merge-config!
                  {:level  (config/get-config :log-level)
                   :middleware []})
      (timbre/info (format "Unkown log system: %s" (config/get-config :log-system))))
    (if verbose
      (timbre/info :system
                   :message "Log connected."
                   :config (select-keys timbre/*config* [:level :ns-whitelist :ns-blacklist :middleware])
                   :appenders (keys (:appenders timbre/*config*))))
    (catch Throwable t (timbre/error "Unable to add appenders to log system" t))))



(comment
  ;; Helpful for testing...
  (config/with-config [:log-host "erk"
                       :log-port 5555
                       :log-system "riemann"
                       :log-level :trace]
    (add-appenders!)))
