(ns tech.logging.logstash
  (:require [clojure.string :as strs]
            [taoensso.timbre :as timbre]
            [cheshire.core :refer [generate-string]])
  (:import (java.io PrintWriter)
           (java.net InetAddress Socket)
           (clojure.lang PersistentArrayMap PersistentVector LazySeq IPersistentMap)
           (java.util Formattable)))

(def conn (atom nil))

(defn connect []
  (let [{:keys [host port]} @conn
        ^InetAddress addr (InetAddress/getByName host)
        sock (Socket. addr (int port))]
    (swap! conn merge
           {:printwriter (PrintWriter. (.getOutputStream sock))
            :socket  sock})))

(defn ok? []
  (let [conn @conn
        ^Socket socket (:socket conn)
        ^PrintWriter printwriter (:printwriter conn)]
    (and socket
         printwriter
         (.isConnected socket)
         (not (.isClosed socket))
         (not (.checkError printwriter)))))

(defn logstash-log
  [output]
  (let [^PrintWriter printwriter (:printwriter @conn)]
    (.println printwriter output)
    (.flush printwriter)))

(defn logstash-appender
  [{:keys [host port output-fn]
    :or {host "127.0.0.1"
         port 4660}}]
  (reset! conn {:host host :port port})
  (connect)
  {:min-level :trace
   :enabled? true
   :fn (fn [data]
         (try
           (if (or (ok?) (connect))
             (logstash-log (generate-string (output-fn data))))
           (catch Exception e (do (println (str "Exception sending log to logstash at host " host ":" port \newline (.getMessage e)))
                                  (.printStackTrace e)))))})
